package SBOM::CycloneDX::Vulnerability::Rating;

use 5.010001;
use strict;
use warnings;
use utf8;

use Types::Standard qw(Str Num InstanceOf Enum);

use Moo;
use namespace::autoclean;

extends 'SBOM::CycloneDX::Base';

has source        => (is => 'rw', isa => InstanceOf ['SBOM::CycloneDX::Source']);
has score         => (is => 'rw', isa => Num);
has severity      => (is => 'rw', isa => Enum [qw(critical high medium low info none unknown)]);
has method        => (is => 'rw', isa => Enum [qw(CVSSv2 CVSSv3 CVSSv31 CVSSv4 OWASP SSVC other)]);
has vector        => (is => 'rw', isa => Str);
has justification => (is => 'rw', isa => Str);

sub TO_JSON {

    my $self = shift;

    my $json = {};

    $json->{source}        = $self->source        if $self->source;
    $json->{score}         = $self->score         if $self->score;
    $json->{severity}      = $self->severity      if $self->severity;
    $json->{method}        = $self->method        if $self->method;
    $json->{vector}        = $self->vector        if $self->vector;
    $json->{justification} = $self->justification if $self->justification;

    return $json;

}

1;

=encoding utf-8

=head1 NAME

SBOM::CycloneDX::Vulnerability::Rating - Rating

=head1 SYNOPSIS

    SBOM::CycloneDX::Vulnerability::Rating->new();


=head1 DESCRIPTION

L<SBOM::CycloneDX::Vulnerability::Rating> Defines the severity or risk ratings of a
vulnerability.

=head2 METHODS

L<SBOM::CycloneDX::Vulnerability::Rating> inherits all methods from L<SBOM::CycloneDX::Base>
and implements the following new ones.

=over

=item SBOM::CycloneDX::Vulnerability::Rating->new( %PARAMS )

Properties:

=over

=item C<justification>, An optional reason for rating the vulnerability as
it was

=item C<method>, Specifies the severity or risk scoring methodology or standard
used.

=item C<score>, The numerical score of the rating.

=item C<severity>, Textual representation of the severity that corresponds
to the numerical score of the rating.

=item C<source>, The source that calculated the severity or risk rating of
the vulnerability.

=item C<vector>, Textual representation of the metric values used to score
the vulnerability

=back

=item $rating->justification

=item $rating->method

=item $rating->score

=item $rating->severity

=item $rating->source

=item $rating->vector

=back


=head1 SUPPORT

=head2 Bugs / Feature Requests

Please report any bugs or feature requests through the issue tracker
at L<https://github.com/giterlizzi/perl-SBOM-CycloneDX/issues>.
You will be notified automatically of any progress on your issue.

=head2 Source Code

This is open source software.  The code repository is available for
public review and contribution under the terms of the license.

L<https://github.com/giterlizzi/perl-SBOM-CycloneDX>

    git clone https://github.com/giterlizzi/perl-SBOM-CycloneDX.git


=head1 AUTHOR

=over 4

=item * Giuseppe Di Terlizzi <gdt@cpan.org>

=back


=head1 LICENSE AND COPYRIGHT

This software is copyright (c) 2025 by Giuseppe Di Terlizzi.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
