use strict;
use Test::More;
use Test::Requires
    'Digest::MD5'
;

use_ok "Algorithm::ConsistentHash::Ketama";

{
    my $ketama = Algorithm::ConsistentHash::Ketama->new();
    $ketama->add_bucket( "localhost:1001", 100 );
    $ketama->add_bucket( "localhost:1002", 200 );
    $ketama->add_bucket( "localhost:1003", 400 );

    note( "Running tests against original ketama object" );
    run_tests( $ketama );

    note( "Running tests against cloned ketama object" );
    my $clone = $ketama->clone;
    $clone->remove_bucket( "localhost:1000" );
    $clone->remove_bucket( "localhost:1004" );

    ok( $clone );
    run_tests( $clone );
    done_testing;
}

sub run_tests {
    my $ketama = shift;
{ # 1. simple check
    my $keys = add_keys(200);
    my $hashed = distribute($ketama, $keys);
    check_consistency($hashed);
    is $hashed->{"localhost:1000"}, -1;
    is $hashed->{"localhost:1004"}, -1;
}

{ # 2. add more buckets
    $ketama->add_bucket( "localhost:1000",   5 );
    $ketama->add_bucket( "localhost:1004", 800 );
    my $keys = add_keys(200);
    my $hashed = distribute($ketama, $keys);
    check_consistency($hashed);
}

{ # 3. remove buckets
    $ketama->remove_bucket( "localhost:1000" );
    my $keys = add_keys(200);
    my $hashed = distribute($ketama, $keys);
    check_consistency($hashed);
}

}
sub distribute {
    my ($ketama, $keys) = @_;

    local $Test::Builder::Level = $Test::Builder::Level + 1;
    my %hashed;
    while ( my ($key, $expected) = each %$keys ) {
        my $got = $ketama->hash($key);
        if (ok $got, "got hashed value for $key as $got") {
            $hashed{$got}++;
        }
        # test logic for returning hash number
        my ($str, $num) = $ketama->hash_with_hashnum($key);
        is($str, $got, "->hash_with_hashnum returns same label as ->hash");
        $str = $ketama->label_from_hashnum($num);
        is($str, $got, "->label_from_hashnum returns same label as ->hash");
    }

    foreach my $key (qw( localhost:1000 localhost:1001 localhost:1002 localhost:1003 localhost:1004 )) {
        if (! defined $hashed{$key}) {
            $hashed{$key} = -1;
        }
    }

    note( explain( \%hashed ) );
    return \%hashed;
}

sub check_consistency {
    my ($hashed) = @_;

    local $Test::Builder::Level = $Test::Builder::Level + 1;
    if ($hashed->{"localhost:1000"} > 0) {
        ok( $hashed->{"localhost:1001"} > $hashed->{"localhost:1000"}, "1001 > 1000" );
    }
    ok( $hashed->{"localhost:1002"} > $hashed->{"localhost:1001"}, "1002 > 1001" );
    ok( $hashed->{"localhost:1003"} > $hashed->{"localhost:1002"}, "1003 > 1002" );
    if ($hashed->{"localhost:1004"} > 0) {
        ok( $hashed->{"localhost:1004"} > $hashed->{"localhost:1003"}, "1004 > 1003" );
    }
}

sub add_keys {
    my ($howmany) = @_;

    my %keys;
    while ($howmany-- > 0) {
        my $key = <DATA>;
        chomp $key;
        if (! $key) {
            Carp::croak("Could not read in line");
        }
        $keys{ $key } = undef;
    }
    return \%keys;
}
__DATA__
6dc1cda6ba8caea25fb32b14851fc86d
ba3272461caec8ec41556596f7df010f
13209f88100162609aeabfa4d7da85b4
13fa74d50ec3cd71d2975e30f8f85bd8
7e0c51bf844b69e2770e4d2f85f6aa73
b30fe2d51f2ce81331667b7c63523fc3
b1ae7436ecfa94a9ffbbcb9acd6cf35a
3ee5e07aa0655623ac15d94b9d629fa9
aff7ea8210d12aa9cf3a3d9c91eb3980
29b2e192fa6cc3fea4fdbf73576aa64d
ee8a3610a0f43d7f2da4d38e51dad493
02c95a61e82cf22ec3d68bf95fa2daa9
8a73b9ac46a808de13f98863f9f5c319
7f147f3a2eaaec0c7d8accc14db60c53
da5aed33f08e0488605b6438f828d33d
b8079b34bf4abc5722c6f0afe1286d2d
55316ba8b906f3342f7913bcb00b81a2
3543af97b66aced00534c1d49f142763
a83855bb9a66ec84f65b2c04122c2447
0073ee1a9b4ba9b4feed48f26b98a092
77ab95e67d16ae5b00e1d1d45d933918
f791b63c743c1277d5704a0c9a1c0586
df06e29e5862c70184e82f580eca7bc7
c4e6a34043256f7758ef87105668d68c
40e83e64838d64e38317beacb25cb055
f73d91c112483c16ac5253cd35f09291
47ad87c460f7116b154688bc1b6bd055
efe9a2a688db6e7da1acd9d08963a2a6
2629359154e0c7b1f005d375a3a461f6
ed623b5692894f792ebe235c5c653a27
7c80af1d1d42a6246b3d47f8fe36ffbc
f1c4606776dd42833a543782de625775
59fc482306f261c61f210b140175b8da
e9ca428d24620c8696bd3a6c0c5854b5
524205160e478686a83a3e93854488c0
a47f9b030da6676b6716033b70cb3b86
7fc030509391d7019f138e268c52087c
ea82dbd8ee048dea6d016ca10b3f1e94
e61d2d2fcddc7a4d217a3ce048549329
5c4bca5772d4748ff824837d7938519c
72a80a26115e283e392a60bc57e2fc34
a2b1812f16e3db35a890300fba3c057e
51df27c338c35740a02bba62d9a30447
0c1146bf126d2275318b906a73a84cb5
02be0fb20a2d9e0bbaf8171260d3c3fa
4925d79c95187e1ec4e37f446884e8dc
37688b5001728cceb9f9baa721513857
49e8d92a3f631b3739e05a075e4b5206
65f4920d7df2945e778abd65a4a2969d
4c9e8591eec75e411879b23d5c0907f2
ffca1dcd3be8ed9c4c6926a3f9a3f012
9a05fda96ca9513751642ad9b5ed47c4
4ed1d04b8469c08eb1db913c344746c3
4a43b66a824730d5304eadce434d4356
f7dd8bdcb8d790b237228eb17d8f2ca7
d2fbba1212026e8d29d96d01d7160071
e55d1d314345dfa50c623f62a0a262c5
e77a1edbe164b82d5c7ef291144b64cb
4c504f9d7f6b33fa92da6c97f6e16bf5
8e2abb9f39ac2d079295f24d4d59c5cf
bf7167360ba8529c778794ee14c0539b
ab64618f3b6765b71ad1aacc1369f83b
8f05c798c08dc316f608d65317c4e60c
38cfd0dd64d589835394c6776fd35018
3ec0274e025483b2349bc4e526b40da7
d65b2b20ea8e7bee4e1a8b8bb7a75911
d332363113f9bc9d4f0992c55305d631
36fd406b77e26c98301a5811a34ce861
831f0c0e394b7d1d20460b15b9094bb4
6c3b3f8d463d0511b9cd55cbab552ad4
e59def80c2474c2604b5f07eee4cee0e
783e1ec2ff006229f6466454b9892021
ec9c7026c5793901e750c4ff2fd85ea6
60d768f77bb3bfa5c8d7ef5b4e6c8543
797c53c4c66b99fb6ddf9f91f5de846e
6e07e0b8ced814eb2c2cec5470da98ad
ba3b8187e6b983169a06a877de5f23ea
2606ec152d13ebb08e9c6875509e2016
2b8c0e3dbebc726e78635772a9602424
10370de71576153802b212828f12915a
e003890d1d2fa1e632a92da067867785
5d0b233ab867d57ec357ac0cc42afe91
bb498350e9cece014d01c32ca255e65d
f4c251b26631ada3f98d5810a725d4cb
cb26863b3e572285de3cd700622eefb8
13ca0e33e9ec08074fc8d6a685ba8a2b
73444e3f7ede827b9db22505ac645bc8
a4c5e6bd846a57661edb0147d4638e1d
133815acd388afea57ad2811fe26eef9
6bc0d0669cf08ccb2170c77aa4dc2e0c
5a68ebb5334056e5593d391618b97682
a6bc335bea6f56f8a8eba3934da9f32a
1f6cf877d409266c41caac2762fa7bc4
ddcf57637cd2ab61d35577d3dd88eda4
266d6afbc167b3ef8ee3bab6124c310b
ec7162a672d24dbeb039e224e411704e
d81f47ebd255cd116630c763d60b6035
90e32abc3df3e1c17dcf3710e96daf1f
3927210d541c829bbca1adc9f8c8ecea
59237fcd9a0287b0116b38326b93167c
4b16c935ab5e616c1d3cbe96711dfbe1
e11130fb371f2b8ad36f503d388cbd4f
d0d4dfde112b8435652343abfc13fda7
c14a667be99bb8fd5f5ae63e88e7d5db
f640b51ad384276354e2333c3067ad51
e2d52819869b6410f0c51fb232905a0e
3bd266b8ae815103ec9fdd19ed1b48ed
377b50dff669bc3b8701f343ff8f2801
7c6405d5f2311fd4bb104395451b6159
e94443c12693fbcce83b57e13c233ad5
8373b5e6deaf6197ee69a614950bf8f9
50532062b858d79ebc3d23cf546de4db
9edc7c01f50323a0ebbd8af5a207aa3e
815037aa8da5dacfa1b8b735356e6ced
e63c3c22a3d39a2b4aa57a27c1631582
9879ad9df1c712cc22ee62e794489eb5
769cf71c382d4c93fb515e019f71b48f
647a96170e8332a4d1298803582be9ce
eb3840b165f2cb8c5dff56b37cf8bd6c
6b8943fdb6c27b9788b4a7de30b6046e
135b7fd997f2d9a11297bcc5b0eaaf09
c10921e85049e98eb1199a65e3cf8aa2
510a91521d06be71f8f96e3c9d7be445
113264429316df34de72f3f17be81eef
ca9bece9402974e05d8232b86dff16ce
194104ef50d5de04c0f32cafecdec6d0
ac6ec03a6cdc76727449c3b2d9f25150
8d924c59f3daa290a7117064a8eed51f
aa653d2c1623f5e846a7794f964a368a
01e0be0dfda075f626a9654c2e903739
651ff19a8e452816715fb0343ea0df76
57ba251f2edd3adfe4b0ee0a6dc4a49b
60e89f1e9a559eb006840bdbf0d8828f
4582b9e44b3c2b1a42699abdb107a86a
8cc5b5f57fc2790c8f17231c4680600a
24ac4ff4c779152950596b8fb2ff66b4
f79072b0941241c8fff042f9af04470e
7779a97f53fca9845ce571806602b2c8
ad2836814ca1692f9b0ecaafc55e4ab2
4a152be330efff1f3309a9d88cc55841
6268a424c8c94959d318e00023c2d213
82cc62ed465f5a6a2f6544c8fc760d43
e88a7f76176b245597a33056d669d1fd
f9c730d83bd64b913c2f30c6aa1b9b86
e21e73f04796820d0317576caf260ea5
86b7758c9feb3264b2f7b29ca19d0b94
002aec2d164d058261cd5c8953bbf211
b769cd21e2011249380fbed12db81f19
aa648bbee11d89dd4da110502c2557d2
d961c1e07cfc7ff1d059ed0c85c52a1f
8140805e06a30fffc4b91658740c3970
54d011f2744c2ac7f18ae99a1ddbac14
fa01965f100b83da74aaae430ccce16c
75dff34a19d653ab1867942394734ffd
9ecd70fe6e1d9a38b6b1c18c0825f83b
df92c72d4819f3e40cee04de14fe01b7
382a004feb8b4e9152a09f9656fecdfa
ae7b59913e2222e351e1572dc814f589
cddfd7304e2fb20943f5cdc063e45b1c
5239739544e08338f43aa21fcbb98dbf
d7f61c9ea7ac3850e2d6e94c0b05b6b2
9402eb9699a5ce29e96bab2f29fe5984
d07ba80ac437febace8fd1e9059a55cd
26edbdcfca40cbabeae2b46f93488482
0efc7daf54d17e4d98c8b2cc0167727d
e1526f04a5f606a0e37efce2a6e8a241
9675c57750a6b3c83d323310a2decedc
ecedbeddbbb114a90629d220904cc4e2
35e4a7a05a32f90273a29d970fa30801
b067b851f4dea1ef4191e69976efa686
a3d87e6c027a436ebb7c7bf924dadc6f
b2898e6aad55bd154cda55020108c977
5e0833421d577b11e6364e6f876f2d19
120cbd9f0e32a6abcb44bd6e2055f3d1
37b3ea297d95739c06966cf60f404634
f665c8f0522be81bf25ac870cebd563e
0d6bdc6a17398fd4c75da65c162547b0
87f0e04953af8f4d2100b530924d99fa
2518efa0e7b2f510166acf414af72483
dfd20660f76d0cc839a2c2668180b3da
727159f2f069fcbe4ec9ede2cbb8af2e
8d26ad38abdec1701df0d49735dc5161
cee80d2e3b9919922f8953f6db578f87
c9035d4900ccebe64ff8cce8d81fae55
0c314844d0d4bc7f5659e1994e677403
80203de6ff26c6b1cb38081bdeccd579
679d57bfebde69e03661d3f467671ea8
62388c7e2d5719b0721a0bd9efeee5c3
825e3eecd3357ee8ad64e2d3cd11967e
2953d43e400cc77ae84b09b4fc20bfa3
6c18df2317713dededc49e83c1fd8d06
f16773d527fdb55b17ce8bc7b734c8e9
089a365b327dfa3890b109257fda6be3
c2b60613bab2f23ad2de3279cc0d71f8
791877b3492a1f14dbfd90fbcc5a5ebb
673ce870626e844b6adc912e24cff85d
8ecd49ed14ccc961f8c66239fa152f70
bf2f352e210c7c1b3904c1aeb972d63f
771123ba38ddf05469ec7408256c0a54
25d3e51b8f98895eca3d28b6b576ec21
68a554d0202b9b13659ef6c38c5c94e4
5a2ab1d6f262734c3e6ea6723867ebd2
b81845f0000044bfff1ec9cf2c4fcf67
7791116d0974efaa97507fd12239965d
e18a421940488d8ed707bce9e4a7a48d
5793694570d844d60cf50f551f125b23
7774e56765072eddb78ad897f2f5370a
a228be28e06c0661b529e5b5dbbf8fda
7b75d0538dde24f28dddafe3f4eb67c3
6625b1cd6bc73e0295ac78471f75c0ef
60f769690688fd1de1c3792e7437ecbc
0d4bbb4b9cfc230c782a0cfd7eaf7846
0d4f6dd2ccbaa94d0492ea6713e7b575
f788f9e5e5d3d6a8c8b40c9a9632341a
3662e03464eb47d0c03b09426aa0a0f6
03bb76c7be739097bcf58a8079768acf
eda1beb9a102e927c49624f5e51ef34c
05582fa48864632c450ec9d40177e1fe
d15a4cf02ccc70a5d8e0314d495b40c4
773d37179110c8a2cc2677cc0c84cd37
c625d7326d1fde145a979d5ecb93eb91
44de564cc03ef60cf4423d8abe1a9c3b
369c5e27bc989847a2bc5e6494db7e98
8f49460dd6a1e7928a21bce168133fb8
60a00c5cbb7a534b0c7803e7d9e08c53
5991020124eb2da0a27a22591dbd84e4
9e575d03dacb0179b88c15b106c298ae
3ee9ec953b0dab55574d28ffa4dcacc0
d383d59b77c0b8fd1d664faec365eb20
a6d203e296c593ed9cacbec3ef3d3219
2c1b42c0e624bb06f5590248e25fa6c9
2c33fe5c71f853952c9fdcf5425659d2
b4d01193788b100614d58eec73c1ef87
54fede7a4156bc139fff8745685f8c71
beae1ad66112c6c29227f53e29959f5d
4b2c34647599d2a4bf34274e53d2edb0
17c6473d1a3537ef9ecb48544d094864
ddd816b7abe7dffc460df56b3183d33e
c53e3c088a4287758dd54c7bd8b15e69
0f1e5157d11a3537a4870474fc242198
2c5d5b5b76da4eda258bb1ec8d5fe357
bdfdd78b5c1cb7b024c11c9712af42da
617e83b485cac7962886efad65906320
cb349d893e015e006af4323f206cb107
60af5a04f87878beff512a2c9abfd6a1
62e15ead07052faf09326e8650c4a63e
0060726ba074af8f2d9c104870145030
ad75f37b45868ce061c0180fb4072d24
e726d533f12c0e8adf2768eaa55faa83
9b4b9fc2c35e00fe342ce5c8801cec2d
91ef9f9a403512a0a446407ea5301933
5d487ef840dd9c9f77ec1f91c4fd003b
03790b82d83f0467469b8faf3e63887e
ec83033fd57002a2a8edd88d44810a1c
de6296f5ad2b9a62a7c3afb6d3e92b7e
dae6bacb459203ab0b6275c69d3d23d2
9c80d783d62cb5d3a5eef9655e92e02e
74f757162dc6b260716b50cf0ef86748
a2dd7c23e89c190485e14bf4ad0f5c54
9b1768e7a5fdae16f2b3b433aa410faf
b9e16d6843b20b4170f79bf1bc1a4992
ef4077f596cd69afb122d0e74e189f9b
1331b6d8492e5fa3ef27185b8637472f
7d34f30a794c47e9e9064b3be83458bc
bd47aa6816e3ff5b20608894a6d7d713
d8eaa9e01c99654b28ee22461c8a32bf
8c42118265297286a9ce1f66ba51e8ea
78243d265b62c24be422dca7d75b8cb4
c4d4bafc775e89432db32a6bca25aebf
165a7ed5ae9b53ded5512550e8bbcb52
e20d620e70558b57317f2d1429ebdbc7
27eaad840d7444ab3cc28897eeff4852
61db3f4f849d6ac021752c5906eecad0
e5f3c71ae2cae84cae7c016524a23b92
19147d5dd853bfd465d5a8c4c9c34fba
d93dad41499e351d8e932bdd41e36cb6
dab6daec1db636bc3b82d994dfe075b8
4591f9f42de6bc847257342777e22193
e84c4984952cb3d59eea935f4deeabb6
9e6f7ece67c81d3b78390075c30ac2f5
a6853b188f6b455299a3a51b7a5bcf3c
e84900c838d0bf3513c050918dd9dcea
d7a2bf90ddba66a7a88e7c2e645ef8ec
8cf41cc8cbd75ebdc15922bbcdca7c5d
34d299aed93729045473b6023137291f
a9ee39996ce94aa9a1154a825e32ba71
688970631dde11598d11c75f68fa9d91
89e2d197746613ed47077aa3f4c4fa54
7bab1daa79cc98f245a8254a38cf5074
fd83465cfc41b26d914583804de6a7bd
d027d3677c1e969de2376ab28593f40b
a54e6822b937890ff17ecd07e341e238
65412f036eddefd925d16b2591839528
2e9d3de8c8c2a84f5ed92ba93807f809
932f3eca3e30c0e6cb77b9e6ba69349c
b99a676d249849dbb388db14d3833d85
d79d5ba2448e1b6b6dbbe30fb9fa58d6
b9eda8e16c276dbff7054bf8e61ae1fd
cb98c6110ec9d2b53bfbfcf63abae9b5
d37eb9ccece9428526f2c7c503b84f05
30a522f721bff308d367c121e824b41c
d034d2b86d761734591fda87575e4030
5d56f1fb4f6810c61dd9a62d2942ef1f
aec0b38679df80e0c37e7157e21b4af7
ab44c7a9b4a6940a316e91d9cdbfd925
05df81da369bad9e0d590b70995b189a
765b9bda086f7d0da0530e3994c63d2b
00d6bc3be6ac161853816f40c1baecf3
d02a8c083ce0dd98c3444bde74e7e9ad
58ab38e63ae742a6979df9b6784e0d1b
ef90d362fc4efa35569aec62ce6aa24b
93c9039345503f778f366a5bc069ce95
b22669a6b86c33deb2eb6bc543557c73
c3c2d9c9a86b0a16adf485c40443eefb
3bddcf641c5be077542b1b1a54f4a688
e27d318fc90d1ffa4cb1fa28c8ef9c7b
6f61112743ca5f9e562f11f989163897
c30321f795122bcd048940f2234bd465
e98dd9090bdbbf813c5ad935f6572d16
794a276f67e45e1d92e9d65417088484
d714f0d536b3d37fe67cddc62bb25f1e
bd989be0d384ce924bbc35c381c4d8da
bae233c3f3843444554c8435f90c4edf
a63833d7f04590e0b631472d8a81b500
319b22fa63a7420bb9aa40a2b6c4b0b8
e1d851291640333a2dc27ce950d8d74c
b653486ff605cd55a9e81013ef4f8484
ccb9ece6c0ad77764d811a36ada81e67
e5e2c35d31ba20de12cb7f59ca14a0e5
a0bdc3408fd62cbeaf1fe419e31defb0
502413181eac3604d7bfe8b2fac442a4
38bc56952ec01c6727100b1982e94104
1eec0cfd7e69a91febdec8d1210f24d6
18f24ffb9a18907686eeca4252a0f38e
85462ee8fd7a9abb7728738dea2a8b3e
a74aa90b827af1e93b33b5452d62b30f
d8aae86677d0d33ce1991b94fcc97c70
19248b7b5bf9953597e247354223d21c
b1c972e20d27797ca184fe08c789b275
1155e7cc34a5b39373eb1648e1244587
c8d3939fb837389a9afd50a1b38abbb9
72f087f0d1d9b744c15b6bac2e8ad4ad
be14e2124af10119d811d34d93404d65
861386f19fbfc2ac81ef7b01df014e8b
1dd0e36c4edeab912e9a873af51949ad
7f358f049d4b156db831427acf646b1c
7331afdc3321b3e49ef7de95abb38b67
4a7440ead8c544154ea21f8288246315
50bcc015c055c3b9f6dee32e0210be11
00b7943d583a7c4bc62f7dbb40b7bc07
9971fd2db74725a22a91dfbedb9737b1
a918151bbaf9a217d6a13c0e8fa1bb62
384b025f3805c091481a92e57a953339
9e9f73fdcfc509aa414adfc8f33d1bc5
8fc1958ca6c22fe4805da5bbef6e4c7b
c13b2c793f7141b46a703a1ab4d71ff7
32c2a27aa99ad6bd4d60f62777754304
93d632b72aa81cc18c005ab25dfe56e0
f0593d0998f721785a36e9bbc6a66af0
1e2d862669eaeb20906628e8558d4b8e
ebd8a92bc503e9f2e757334b9ff66f9b
4fe94b0f155988552d05e9d6b780751d
54803e5af07b8cbc5398171cb932fe8c
5a0d6c53b99775214dc81e4825a6f795
f4bec6518f1b4e4bcaa527d90d63577a
81c3382e31c2fdb02a67777371b1d2b8
12e80df7310ed6b7a8b5b152c4d9037e
0518557bd2eedb6dd8f5ae5881e95f95
69d523f717e0695b64c6c4315166a8e9
69befe58fb637eb5f43f376486232fd7
86c125dd41fb435dbd6fec683d42d9f5
94118eadc9a89de7737f1483134be188
d0b3459cbbb959267b0a43aa01f80e60
79bcd8765127537c41f5608730836d90
e4884ebe20f97b83287bc796580e4966
54e4f4ce8e7fc2816604547d968c63ce
7fd066e3ce562402c2b80c511ab6d6e2
d6fe4d8f9a15531c23c22ce178e15c89
eaa6e588180219fff32ba66b2b7407b7
69a119bea3989372ed9e9194cca59f90
7778cb0ebbdfc545a5bacb2608b2d660
42390a1e65bdfbda6840404c110e03db
687d5773fcfc54fb1e829c841c255870
309e8c85d142e0681f779d7c09face16
f88ea24219e3c5d41c16c354d4b030bf
61c4023d47d9f09a0be83a5ed8a01ab6
990a1b3d1914351708207e4c47c11324
8857ddbfdc4fadc8edc7556afc18932f
f1bfe55eb0ab230ae8d0b9b2f6d98671
4c533b29db0e6b0279aed4951b564f0c
7ea878224c3732e12941ec5e3d8e9508
26cd7d0c0713fa0719f4bb06cc5f0a7b
8f42c4fc4e3fdbef15da20fea29b3918
5744527b93ca691cb9814c9c5f0993e1
e62a30401c686efd997b8e88d0a7c620
846a4d881d0bb22c72ff763760b9660c
33e063c1fd61c92cf227b44c7b6aeeae
605fd03c3864de0a024b92c119e3de6b
3a3c6adac015d8de52285d5934350f83
bf91970543fa59bf1941d49de2cb1fc8
5260163aadb2daec8c07a7d193484513
6392bce21434c0bd6c98a9bfe339221b
4438ef2ba3b0d8b0018ddbf3bc44e037
26d4e6ad2a5d585431346b417264269c
d2bc952ebe449c0b35d1f00587147be5
57c0a5f65098189b46a09d99e45cc871
e79f05dd3bd0da17f61d1a9c4b095580
fb1fd9999bd55d9eb6c25c67ec59ce69
834f6264799940371d19bd50d3106ffc
feb41a07dc0380fc2be14bf20de15802
a513269c6891015be0d02aebc9fc3c6b
fdbafc6ddd2488e2453f719213059d4f
65381f8e82138a894c35ae985ef07d4f
e5786c5f8efd776cfa507371ddb4dba6
56160c6b2667d854e4fa1fe5b4517de0
0c15649db90c06d6d7b6d04a857ba874
e316cb2539df96845033579735e33205
0e2387f10d59eee2ac3b31489f01cd14
81b0583ea0884e749aceb1149994bb6d
c39deca5eded7b06d69cb60298f23a15
8c58b3192a73fb2df1d82a27ec3b343f
a7ed9ef5456bd0f8fa7a0ec1028bb819
64eb92c7f266fde5c9cba4cfd5ff6ae1
5210d18e92596323f651dae8e4d019af
f8f71dc19f7b53367a427a2a52975670
7fd365934b1cadb99b3841146f2997fa
6ca41ef7b11827e682bb2fd452d0bfbb
89f713ab8a18f751d5784ec7b51fad11
95d2fc458d0497687a47ff2a379d0106
2f3935c7f6fc9c5165f773dc1a46e38c
5e1e735e9b1a4b903cbe635659b182c9
849ee96e144db23b412dfa695d5f814b
378ee06dfe4b4517892b9f448b470de4
37fee7729bdc646a538b36dfd50e976b
e055cc3708159a433a703e9edba27e42
5c3384a5e36e1510438d73c9b0518dac
bc918d7cddffef67bee07ca750d21708
47bb315cb385c95744213d2b5aab9d3d
d468abdf7afa7170f1a244c2c255f40d
8d2bd566312ea2080d4221a22a945509
9e3367d4d2dcef365306fdc7cf60d8b9
995be8e5c8e2ee6531ce9d9aa087130a
c8687d794efed1da154970a94bd90141
5ac114ce41512916a03fde0fd739a52b
74d015bf2a9858575a9fe19b154702fb
3bae3e1fdcb2e19ed16d5271beef4dd1
ea3a679ed736ea8b34e069214e2b0dcc
4f2a5dfaa22e9f419d0fee37f922de5c
efb75465362f85919e0aeb4aac6da9d9
ceb1e24e8ecb29f0d5ab3ea8ff79a1d8
17badcbde57013d0533845be0e68aba0
e39c45e54d54eff2873237737c1dee58
eaa89846dbb7f5f1896133cba156c473
047e76f3454788fe47983886c43ebe86
167d7b2dab321df4b5d19f288b36716f
b443ce9650dc4e45fb1fcc8a2b8f4579
b1ba65eeb3189b74ba1e59ebd6bf26e9
b0165af03a95847bffd6817bf8abaed1
45452065355f80c6777833b42243ef83
8495bfbb563ec963b6937d91f70bb9a0
3a8b41a56ed608baa2e3ecd2b0037a1f
6337f20598049993ae8fb377563c2ad5
c94fb4ef2176093bf99e4608fc308d6d
78335cfd3cafffc679725decc63fb054
b5f80725ca8959e8775e866c2f7591fd
5991d82a38bed684bb2a137959052495
8db74ea9113893bd10f3f54c1fa6a609
73dfd8cb8be8d62e25d6ca8a88d431b2
d445c05c62925d001c8a631e8c0a6e94
9e316d1186455f4f2621648f09a72d17
c7aba7987ea5b9a232340e295087df5b
a66acd5d284247524d089e11ea62cc88
ce2e35645fcba46f1940510ef843845e
fd152f7146bba6d56378672e259833ee
6ff30eb294aa37abdd8c1c38fa8ced58
f7f335d55a246cbcf3eccfc53c4f2869
6e16e2bf817fb4b1247b30599bfcf34e
eb5eb0a5bb072ba4d78e68fcccd96e8a
83bb67c815719a7fc587a8fe4578f686
8c18fc5943a17c08ed9eddbec1ac6987
2c340c66b13eecd74742184a23cc3364
8e9c301c9ad198a32d43d98a98a3526b
e1bb6d77e6e0e1cdf32844b48a653b45
4a59fd20f606705194000cf0c6d6ae3a
388e5e8016040b87ab298762ce12edf3
97508639b37ec1c09d539edd793c8b87
6b965e3767eefd1f5605b2e916bc4204
fec47fae77c7fcfa9923099ac9818f9d
903eb38d647721add29e9a85caa47d4e
0ddb8e0e891c44c2beec07da29d7a33c
41738be3a6d0322237a4ba7c5c3283d4
72f8269f9ef8b8a5bd82b3a02c6f4452
64f1af706d4710dee22d526ffc45da00
11f8b587ec82f48f3dc361f904a8ff73
c033fbad00c45d1eaeebcf92b990b77d
42d82f593f2879d122e5a8a8e5271787
2e9d2b38b965a7777fd1925501943586
1ab91d1a4cf72dcaa2d68e6f4a04eed7
338a04ce3b9a9a6345ef6f8412462d87
272aa105e102d10ab3856fb0ca97cf8f
cfb7959139325d45723d7a74a65fe7ac
10e70f6ab0a37542bf126c241072fcce
bc500032964132d24d93a3a8db49e9fb
4901e804d67fcc91c9f383bb9636395c
d4a485e477938488bff9600bff69915d
961674cb0dfdd7d2dfe3417743b08463
3b11b1512d030f8f1f0c9188093243a5
8b4ddc3d2aedff731fbfd49a612a7044
cc43ee346f4080cb208b130c3693d796
aa3a670339d6a33cfac5fa2af90432dc
aa7b3ef2113f5dee2f9f4d216d3c12f3
6dc88cb1e36d749a0ffc58f6048c7479
55328b7f1172709bddb6870710ff0118
64880906748a5aba1f59c34ddf3ce16d
5dc74fd9ae9b8880369fa8c979a866bd
e069738a540d354afbff07031de88515
8ea484ee6bf506d2a8dea360f9946006
db37d96dc8c5ecd7cffe5351861735d1
15b46b0c2dce76343d2a7b8e616df6b1
bf0b22daff921e122f0bf72a6479395a
e64062dee1953bcb1e4fb1b43e9d6ccf
665945013af2726df31d04ca8dfd4ec9
56e1adf88bc074f3b5d252b7d0d298ac
bcd60f71e861c40a6decf01465a27813
598020318b372e995ebdda44322e66b7
019ab9a82b57723d84c08f3564e59ad6
e2b549a6252b85e502d93e93d5481647
f5713ab7d4c1409d276986c689380e5b
e5f06f92020e35f85321e1ad605cf71f
7d4172a6cb366c35a904fab03ba42dd8
b7f760549a21e8c88aa4815dd32d682b
e9066b206c2ee8fdd18ebd22e5a6bc92
6fc7bc13aef09077cec33e3de2ae083e
f4d66592a3568f4aad539c2abe8f8007
7d91e9f414f2ebdddfff6cedbfb02cc3
add5d4b51f6b55cc00a4fedb9866467c
dceceadd7af4121c61a1fffa63dc64ce
bfea3767fb326ba49a11133373705f1d
393344e0d19a2a9103d4c9f2683957f5
3b0f9bafec437f029c983439604aaeea
94aa98cd82b64cb1ea911010f6f1a11f
00a32e703d21736e427f54a3e3a69311
a2d8795c51129fe6476706b2425e0179
51e73e1fb5547b83bbcaa1bac24314ee
fdfa03fda56e525538ea8a89be42e3b4
0303649b7a7a5ab2e6191e8da1984f70
eb23f1b2de22c24af3d110f03d8afe69
437773aea533e050305b2a9d847ff1ae
91320af2b01cefd8804cdb8705afced8
135f8847f59f2c4cbe8f9221a0ecf0ed
997479593c1c33fc6033bea162bf05a4
e7429a0fd7661a14695d8b6244efc798
81385881e10f2364fa5f4824671ff1f3
3db966b7c0c949389d613cc8a3965028
d3ed19574bb1d556df7f60eba44da0c1
738fe5cae8c014d2f9d3e34be8017587
b196d08debed8b8be97e2a9991c41b62
ef6ca895c92a08bc15a125f71e69a958
9cf5fca760b25be0a36f0ad4f001fa08
384fa35dc8ae1ee11045537e785746cb
d7a57b48e1ef2161706e7aaf200952bb
3084f84888128450b42991de322dcd33
39287fe32783329a7b65dc1786a2bbee
831f5f104bdbda885e555862226f9678
859ce824dfeaa0c83011e37b5b5171fa
8d12d89a139a4e5edff02047f9ef3fa7
eb5daebeec77e5f71f059a70666ee174
aab658b6eaef65c03ec348a94e020a01
8de4d0c2d13ca143733bef3a8659792a
de652475323ec81eb502486cd1fe4ceb
d328663c7693c951dec3e117a9eac667
383066ce7908750899bf3438b6b46d12
1ae756d1fd1dad8d73aa7d9bc413f407
8a7738c89c210da777a61fd02c3ae6c2
29b2af25799edff824e5f6ad04a16c15
4131d7672ccc781e045ca02aa2dfb8e8
d36167e2c54d3a4e24157385a30d8dcf
b646103a2824753ef04fb8c5e7ff352d
0b33c2c14f53aa8f4a21725512fdda4f
b86f3f70c928b4dd56e748146a2eaa23
6792ebe4eafb29b2ca599582bcc1078a
3c95e05860c49bf8433c5755f31d17ca
249ea9c5ab33ac096bbb605b107576b6
7e7d6128fda17e7851fbd29259210db6
76232ae8b4aded575fd172c46e057ac7
997d75d73711e745388e03c2d11755d9
534dbf1a30ac597c3c9a0d4be255046e
f004e9ec172a2c3318a0f838d3d95067
ce1162c189d631cba69bdd327c8ddba4
457bbef05dc827592e8b1a761e00d8b0
143eb0767254ae15783ff24c2839f288
d59ee83d018a6db0549de804a3a33d21
a35d41848d9974453454f50217685074
300881c87988acab403cc5f91f79cf73
7f943e589f1ae08c55bf9f490fe7aeef
d562dcaa563076f04c33246ee9d3b5c3
24f0b5ed631e9ffc8ef9f043643c4b24
36552f1428f79d01985d3d0ce9e1688f
14915e7f199dbf37d652c695f7b981a7
e5a17cb6b668c0d16b5b84ad07c866d0
a0953f3c5091c6c2fc17d666d0bb61a1
adbd8e749de1d9362762bc3afb3420f7
7d1133445aa0ed76c585ba323b674802
b930479f21bfa497e6575ab78085a37f
85ee408b12f708d69f6f8ccbd29d4b8a
5510b7ebd8c84f504cfdc1089abce2cc
0840ffd956c82410a9c27b8aee1c96c7
07c6d933979377cc79a9e23237e8ec8e
a071e3377f08f9977da0d8f263bf572f
d66f8111d49c858155a47e1cce171f41
0d5f8924e4a9513aad185f553beff73d
ecb912852454acf6bd8a0e6bf59f8b29
b130053d09ea120545333f952e040d16
895d3f5f0fd290eba62b1150b2363e58
83c9f11423832e88d3cc36752a485628
5de6ca0da5b2005bab47e4ef713c0600
5dc49767277dbafcc3acea1762098b31
7ae0439e969a5a628d64973b391ab4f8
652f65205b6cc083167025bd7376fb9e
a23c930f3562290d9a3cde406a401942
d05065687d9913311b00d001ebaa93fd
526e09e13acd742d8570d0014109fdb6
534ef987093b3d7a56af1908c5993321
357a75aa0428bec8080e293f86bb3064
ea2a7bb683002d861f8298ab6ddb472f
3dfe27ba8f40926d934684977945109e
eaec3d3e1087d1a16f58265653497c0b
b50ab7aed8bec37646869eabf6fedc78
786a311c66e687630f978a5d43a4b9c0
d04be631613592b78bddc2185dff057c
e96609bdb300ed185690cdec03f451b4
86f2999218da6224ab9768a5fdb7fecf
b93d93163f0cb6435931735757d6eeff
0838e8ac16e7952d38b5803839125d0e
daac45ce5326ace60451d2b0a1689ffd
fa2fe7da918d6098093e71085cfecc4e
db35c82e3b861728abcacbe1d30fb9ca
cedb2864a0d31014d3717cce2bfa246b
e910da2e72c70c48cdcce7cd999d145f
6f192de4e8682a6dc7696c840aa4cbeb
b49e8cc7ddc16a781d541a4901cdc773
e04d9f073375918e2a4147196c3afc4a
b9edf909715df8e8b982dd3a73529c22
18873d8f5937d6a73aaf35eeb1bb8498
3a1b5c8857cab3b4165387ab7af55d92
37bdaecad5ecca81e6f3f58bb5c22010
3ed0a2315db28c6567e00b00c99c5a43
1b8fd03c2fb0b70aa83468cf930107de
fb6cbc0678c0e6c45ad406a277abb60c
ac6d1efd6de66c7de496bd4066867074
ac3c0a28fecb9f2f35a954cb140f963c
8630ff3268c6d3040d1154d69c1a325d
959b0e1626bb9df479be7df935225524
9ae849f7ce4957b5f18a8b1db13a8a78
a9483e09e0d28e712ed90111257c2f4a
a6e5951df2b83822e69cf1d3116ae09d
3c5e06e53f31856a329d4bc0b1f65322
6793ff83d5d61e394733d253a08aca57
86b587bc7e8ad19b7f1cc9c9ef0d0b4f
a2bc839c31b1f2998bf0139229a481e2
299079034503c6b50a9a8eceed1549bf
2377a603e6229ad9fab5bd3e4341c171
4f89f055e40e99ab974b60fc1da49d05
e05f1a1fd8523efbe03bd375ea600883
1be27dc8f6bcdce5fc4607f8045193a4
a67de39c376e597794477540a0347b39
d0348e5de6fcb512b2cfd30cf3c0d7cf
30806d32d3c476619ce8646a6ce6e5d4
e3ae3506baa12cbf5cef6607dbce57dd
8165b7722aaaa2748b996d2eefae1bdf
92ec81881e6742b8ab1614dc9d7c69e1
272587b7f532da0f0da287fd62ac607e
8045053960491c2b6b85c3d6651cd0e3
5fd70d908ef97ccead04cde7d9f08865
e5fd95e08158ca63f0213a595d2c724a
2be88d6f756f5e4a8daaec3b53d474e9
4f26b0f5388ca932db5cc501108cacb6
d0596c4c3dc67644bcd9f6ba0ef1e928
3fdac5be3f9c5b9806ef2dfdc6724d7b
0c8f218c31f65c870b15a96046ac63a0
1d17f6b5a97db3727959ac717f710fcd
7ef42d0722003f8bf12f8852726c85b7
ed8f03bab2eb1488ee0ed95f2b8be058
cca6e9131ec90d2c89b814eb05d4dfb5
ae78afced11f36f5791d9c9d862aa396
2d361ce8fcfe7f2cc149f27c75c54c77
433f20b66b3aabe10390325838225b9d
c87aa101861f22784debd2213f7b4983
a782bc4987f106d96c66ecec90acb415
8e18e47d4a4cd45e65a269792e69dcf5
ba78451e81ecdbdd5cf5fece28cf0d17
36c3648811c94c82ff7fde5ff49a5b43
e7268b67a09e9fa0a003b225c5eb90cc
b92b4bf3312a3bfd8846c5c850ccda81
5cfd3c4e97bb222665453a814fa985a9
637c524e589174df32efeee48f2379f2
36bbc6e67b131ffe66dabd71ecaca09a
5e104c75ac3daa798cbf6d1282cf6ad7
345f5be52c4a41e25c5fd319d92bc529
79e08c44d925d807614397ddecf042f5
ffed7b440f817823b1ea3cb18cea5528
3b2c6d13f7749fcf55a6ba4940c1a6e0
539e3fd9e5010db3fbc3b8f0f66c0d51
8c841810f37cb1bd8f1c9904ed2639b1
a92d2c3f122e5d20c4631e87573dfb6c
50338a95695dd3997a90bde6ef7115fe
00ec7cc5bff563f9dfb640f34d1ca26e
67723dcf7bb0b2f7af0e8f751bfcc9a6
cb416c57b266b526e8111f060fb636b2
c4cf12730d6ddaf7b39559b5a5e7b1a7
c02d69ebed8bd71a94850e3cff9d46bc
a713bcb01066f9cf42d26ba14a80f441
5bb4270a831c30922e2620c05c2910f4
d7954aebc2b13f3492f7c5bc29728534
baae1cf831b1d84ef7dea26364fcdb92
a05318159f1a19a94be6e48966da6be5
a9652b271227955eb0e6f3191c5d3e90
e2a06701b626ffeb2e98ad6ad0784524
7538195e972718b0fde0ada2324e6af7
31ffedea0b3cf4802599a6d2d9a1b9e3
441069a010b132560284e486163790db
2f40e94a655e67c9230dde06aaca7a7f
3a9a341aa1e0eddfc9fa20d090045848
defe65292219122821fc25e05eaa182a
7623dabbaefb657eeb048dc9f14007a6
1ed4456918e8805cd4b5c6b3075f7f35
2a0ab45e5c2c600ad430afa45b151f8b
b74c519e5e366e3474b72371d5ba916d
3b9aa87d40d923169c43f0b19f222d9e
1daec5b40881874effca597450384532
ab6fac7d6daea52638551f0a904c55cc
4a0a68696ecc49c67c3091a5ff55185f
a889edf014db30bffbd828d1994dd73e
0c81201539ab8362e9633b81de4768ec
d45b1ac0911f96e34e6c9204e5368526
9fc66cf95507089911bb657761a45d79
930bf0b24b6a33db2d981dfc40ae8656
da4b7dee573ab2b498e3eb1de473d875
e7d6081a86a654dc16853284d35930f0
4ac34eba006db53e952fa3a296704f0f
bce1b7fd9870b2c1afa7e8484976576e
fa6a2ac561ee1b4dd5eb7b0439a63685
d279aeeef9f2b87bf22b9e1c4dbe012f
e462bae0f76062aac89886fe41f35ef2
74309bb044b1fda48b6814daf2caa39c
f9f01bc77f7ceeea914fe660cbf964cb
302a77954bac334699b474b80222ec99
809eedfe42e97ab26070db5e3b14aad9
c440bbdda1946399d85b945aa65dbd0c
c8874b2e564c33675dfb9aacafa64de5
6786028055cdf8f73c412b9939d59ead
a73da1a33b978a16f71c6046ee38d092
7cffbe24f0a993973fe523810fd2936f
a67b127cddcd7a8a7f791afe73041e51
6c7d79524358983a9bd076499bf81347
3634df73c4719b8e7266b697ab69ae74
59e03fcfcb2f177a97f0dd3140945373
ff78c29fd6c1d4c2bd388db023808632
121f1c231e137d5660c25229954131fb
c09e6aed5d4ab6044b9be45aa83ec9e8
0339ade3bb496c042ddf3ef6da186770
356d058d3ce3aa34d488593e5687d440
14dff43726eb5f68b7bc685f73e55a4b
9729ad5bb78fe7d2b2ec4ff2c6b3e34e
50de5c6f32a146956120a837ea539618
2d4de278b3ae2e6a6a0dfe8d13c18d1e
fe6ca13ebf8c9f5112a6c243c07cfe45
ca1d6c9de815e89abae0fc031bef8a2d
778b12a891e93a259ebaaccf7c261abb
a9526d590dd806f5402881b3cc186902
be516138865a6533474f72241ff2be31
52f7f52b6f740cefca498e70927ee869
ade735a9bf3b79076a14e6021b738940
d7fecd191fd05e522a2e23a5d03d4ed9
2ef64b010b18f8e2ef544275277d821a
f80de71aed2ecee5dfd086b812987372
7ad85d29fa429dd0eb4f4ea094237e7b
6fdec49b0275a82b6fec8f8947ad2e14
6d6a3e95129f82e177fa73db0c62ffe8
698190ee2d7cfbb50e0edaf3715e9284
47a7a1175a3a5049561dd7017e4418cd
d5ca4438922a3d2c1b309076427002c8
41698680d8ceedb948c775c44ed6bf9f
8b1ecf5abe9f9433d4509ce33e21cdd2
758fd87c2aee4a54398e394557ee1207
74d11445066ba5b1da27d7cd3ebc1268
935c24b40b5795f5a94654f832649684
67a895971ed5af4b137e8a3f9d80cc73
57710f2e23a4522d9b38983e5196abf0
7b7c644e5205c5f00d2238f1dcc1341f
1ec85ab6e0aed0b071064f55b0c05a3c
9aae35a05b4d3d63b13f3a1f4f9fc0c4
16998be17b1197951623e4ae1c836d09
15e2e52481b05891807ef2d48701a047
dee599f75e34f46cf834d79b5322dc6b
4b7d1bb2b0d6c4e91286e052aa54b430
1926d9c6f1db3c7cb9ad646c355e6d84
e806ed83adaf7cded9275007a39810a5
d7b318853088b3492dcc8cd02e6d2609
cf2455e206f2701134115d3db38cd792
9cf54dee16826cc8b50e7862b5fd5c1b
d2d351a71bbed8f2718c2171b57b1fc3
21f6165afd55b441156f694bdfa14ada
43c4d146f5291b3f36f051050f4b1a84
0376710c128da019c8b121ac5f2003c2
84f1775660dd65aa557e83be7095c58c
747a88cda1275e04970c44cbf9598d30
49d8c5aee2d8ae18311ed2c335b05bb7
5f44ed39775faa6c4358b2deae34510e
86356cd2d3cf6a1150495199857cc42e
a1e1a2b5f4cfdf441a007315d628e3e3
6db0a7760fe1ba60f33d051a56cd7b25
856d7316135cfc806da6a134144ce82f
4a66fbe80e326cef5e3eb9d2676ca4e6
0e5e7fc4ae02f15999f3d27149289c47
35a9a2256ad7d2b6f6d2556ea7cee93b
400dc66b32654d2cebe95846a9ee8580
c1c39a5eee725ce0afc53ddc3bfb66ad
8c1fc0660534365507b2c2817607dfb8
10ad2c6c45f5c132564a6b9898ab995c
13a2d52725789a23a1c778f0dc6931e3
e15a247468a9ef0b3d8517ed3ada6c48
3df7bc08f76d8edaddb852dc13a5cce9
304d0bceed6a9a50f803c3390f067d61
054bfa711e81afa63a0e0e99143247ea
7877d83d4c1f5e305334f24d1c9197c8
277a7a19872a7356145d3801a7c3b4b6
0afb1282968487a7a4690f29e2b40efd
83292a2059f484d757fc7891ca1acb7a
4cad304e38025719209c2c97e62bff0c
82cb8d705b4c4c19957f5d1da4c1e3f4
1ff62b4db1fbe0638ffe461f9136f491
b5e96d21cd7f46b5f4e72cd92e1381c2
af1dbb54ccb1aec096c82d06217ee54a
df2a8f0a205032ade5cbf601cdf185b0
21c3c692508cdae62f70525408c51d4d
d4afb0fe9c3bc046b5e55e151fd4df49
93e0af163326c4cd0fe00dc8c10ba46a
1c12d430cd4cdb7da540af97198b1d52
7f8bcb09bf282970397e967936930e06
bfd66e448300978af2b2cd9a8e053f44
5f304b74aa452b51c2d2bb0d9a1bed3a
64603c0cccdc93939780b8aaffe8003c
04a6274950f7daeb4f6f296250dec31c
e637ea9767697282b7fe1510dbc1c2b7
59055585a1e0829aa63ea337e1d0ff2f
39df7e14239d8b86827e498aa83e8a2c
a75ab16a6f988bc8e48e55a6450bd619
9f6cce4075bcc944f3244f30ae4e34e8
124edc05118c7b6ed27baadaa8815aac
2ad451143e6f0a5152ecd0a494aee439
744fce003db9b2f6357ec3203e757703
cb25b048a8d1e91c0513bb7e8daa62c4
0c3625f0492177f703261975fbf26a65
8aabc596699cf6ddbee03cc977c717df
cf6680a23d700cfe482e87c1a25f9baa
66ca739f5088a3fb9a28b18f24742c08
eabc723eb05476f042592a9203808163
a9f6938b8721835ae7fad9ef6161f497
af50d1d251ddb25c11462d87cff6c2b0
d2654fd5796afaa66db88b258ee98cb1
dd50e0b15f8b3aa3d56ef90fbb2a5310
59c32076f508835c5c67d6c3702d92d9
0949f8375b2668a6f9acfdcaf6e91c36
c9661a85bdbc886b92b81d192f6f63b7
fd807df84403ddcd81c4117bffb71999
894361e398e09ca0e8d09dd2d6269b1e
a1d1d183f510a8b6a2d9bf307ae3ca20
ccb7efd3be55f2d837e273d170499c76
5ebfdac03cfcdb6342c0c0672fd44ae5
c6570ff73a9ceb2d62958c22370197ee
1e05641e3097e0be33895afb23384b7b
cc61adc64470ee07e6e162e40d9689c2
fe1f862aa3461e3e1870e316b33ef546
58597655313b2821449681613043212a
de91752f11f43e89d1de98ea2523a266
0214110862118965531df018333c2f4d
049e0cb18038be2a5ec80eccd3d92f0a
5913181175cab7e215caaf13fb94bd3b
45da3aa5112e2ab69e7889d11e349f00
4436af6801d3c1812a8823c5eb0c3f58
f64a61eb3b45743fd2c6575af220ca4d
57b557a9c324ba81e51c200a0458e333
aaeaa18b52e256bb188c26f6c1a318a5
de23c5e306fa4207bf2f30886ec51b5d
6c32423f3f724717f8b9e60713510f5f
708bd3acfef280a723406ec70b01a76c
fde377a9467d9fccf8ca2c42d97ed5b4
5eff66d9640c6e00d5322556fc0bb27c
647f5f0458278264166a823811018e19
839bdcdc96e541939ca7aac4bf3ce177
b75a54865eb5e2df826bcf54a989895e
544a666e8227e966bec157d52b32253c
748e8aac31faf2febf4b3d3f69d52416
3c39d86824d1d6908508296b1b0c2426
3dc3b95e3910b31a848b33e185fae806
0fe138e1b20dce0f1633465fa0964c85
5318d8427e9ab8a6d55386839321a479
d3151404d32a5ba11fc7bed5f3479a02
2d89180deec8b05d97fd801830ab3b47
1510b73f80fa48fb1269c8b86183df4f
e61b6917fe484fc0d859c25c0bb266ea
0efe2b6dc46765fd266e74d5b3e14056
3ebcd7570560133fffb1bd9136ce6b39
17fd4f7ffae7c61a69a9587595bb1802
c9938f19d5f8aced634cc2a045d81a72
56e5db1621b6dbb9c3fdff07b2c0b6ea
32c65890bb1878f7e4a76d2eb65f074e
0eef126bb8679e8ce36041c4a84d8f8f
a3c8b6cbfb13cdce5ecee8f19a31c7b5
205a8d6f26b76628f82bc7b544220b66
b35590b39426fa357be5e82f8e4a5d2f
3e92545b2ea5e39bf64296dddb1189b7
ba02ee687542a63f259dad7f24535f40
4441d38bcadc11e153d14d9a9b7465d9
9e14a0ccc2653c4afa13ada8eb7d15e8
1db4b70fefd7b17f0b956af985f8451f
04b026130cad510c1b3f88f555966542
001ff60a3e27f6a9b6d78b78d01e862e
fd090ad61ced34606d21e574b9214a04
af147701f1fc05bb8385dd93a3709b8d
9668894274ed245e261c8326dfa264ff
98ac482101de1bcbafc7a5c512d73c9d
c88b7661e5a3d5123ada2e12b108689f
1fe507ca40a3dbc4b0f7305c9346dd6e
653846c7aa210b12007f45cf184de431
12595fb475f53e10a1bb3c60e51660a7
e05283943d1c7b8e3e023d67a5ddf0d2
1c18782e463cda4d57a87a3ecb6357a5
f74eb5895b530f0a06acf05645ce3944
615448bedec291248c3734120fb37b03
41ae00a6a2e4749cda6d03797dfb5286
c064411c93a2ae9404eb297764be1c06
555b3567ebe3bde7c0cfcd9a025e40f7
732880b7776d29366d7d55ac9e83d121
5f6ac101a947c0b19c01bd35a3e30639
90666cdc50e8a920b351cfd7f0d56e3e
9c5adc010fd3f4bb75018935dd576b3e
e58a683e1ccfd1e8a9f662f69f543b75
9c6b91c525d8c3def27c2efad62ba59d
bceb8fcb56161b215126514f9a34ca0b
d4749d4b4a37215190c9032b78724563
ca9cf76606fd107a7d9496b9fc2dc212
5fd18de6bbccf47750c4a366b9fa3585
4d9adb0d3bdd6c26f1b1b6a4076a39f5
ba30a3f728a44b4501199bf047a51d2d
b268468302c925e9f28e223d57102f1c
2782c0993a2b79ac7ca24ce8b341e0b7
cdd862c803a9abd701d59682388288be
07483b36d5a7ab229db8bc509f820e79
e4f249b7bf8794ddd41a7ee3640a70f2
3ade7f88ba6550b94ebf7e25939da0af
ebc080c377bba1f5e8299b0d1d0e1607
af19a7bb632a0c2276e7a9f8d11cec9c
e8098eb63ed8842eb945b29531a71fc0
ae2f13fba12b08c9c0f62b712576e175
59fdc0c8d7bb6856142c2181de46328c
1ed3d891c7779d90e4df14f59025cedf
27f9912fb6f714642ed340aa05d6ba82
5c9bf6e78b8b63c1c7c170f2eb763e91
20e4d1753b054e81653e28ff8825f7cd
b9afc8ff94c61bb58039012d19f11b0b
48ba396008e7f30be60226856c3b7ddb
3778c7414a282deecbebc562918147f8
11df6a383c74e9084c565ab8f4029af4
47b45563789c4a4cdbeff2a9acad135b
755e9047376a10c7f9629f2b12a048c9
082a4fd0eb1415ab29c290109bc3f2d0
6182d1f47e35956421edd292a40fcfc9
e1ba034ff5af4879408bbbf4fa2c808f
f161541860b55b20f98271923772aad7
bef7dae3389e34a6f0e928b442de58d8
e022dcbfa05f2736a00a9d55193a82fd
581cc77f53ed2f6fd179be6d6dbe0132
77bc07364c35abc00a9c91f6b66c6760
0f373871ffd3699de550dca1b9723910
95bb98308e13e58a6a89df9da2012cc9
b82f2a7725a13c0351ffb05be8db3f0f
0747c5ad22e0155534fbec8ae3dd60c7
d3a47d3b9e5be5fca3a58635303c6979
4881b7bb0199d87a8e626e87f92d2669
b7aeb22da49ecc095477b7fae58c51bd
a2b1aab292869ab03e213ac899f60306
ffc18988681af0a8d76b0beac0e21190
72a037ee3459b5cc4a3088edeed88f26
cd43733021983c8704901efc017a5bef
e915fee02f264d7aa6295c3cec305522
de7db13ac8ce3337076f82acc4d91dcf
275d509562d4b6a9bbdcfc600e126528
5581d0ae383fd48fc1d1d2503193013a
024ffeebc5b113220819151db9e54380
6c00c29768aad2bdbcb697754b0d5c86
693ee828286a0052956cb513bd524247
0ae2aa136786378bb4190af7284029c4
404c6df0c7b5d56f9a8062b79dc14e14
220950e3acea44e19310c079dda65ba8
ce462a55ae4976e70cc380e0a8be2538
4c967786c2736466712f73d55ec26003
1604df88b457dcdc0753895f21535ae2
53e6cad330e60444a291287596c5fbe9
0928b64911c27aad0bb8f103f6477de4
8843c9d588718a624435eb68e03123b9
3fb3aa2467cc29da0969f9d6518c4a06
c8b9e517bb9e1bb2d8baebb338b07555
f18e9b219f0eb43c51d39ae3a0d54175
b8ead435ec98c3feceb6b3395daaa00a
67f7017e16e5de078e4de30121b3f70c
6d697ac56f29603c0143fc34618050f1
e8bef72665e30d5db82b5cac4c21e3af
42ceed9f863b1d30a9359b64bef5873b
b13e1bd913918cd5c941f10d60ad7e48
31708fd977508140ce7222e4ede7588b
66afbaa9d0f43ab0ac042e5fe19dd5fd
8ab17d326d09e8d6150077d4b58b446c
4bda13987d775c143adb2b548de99eff
4bd687af35de18f507885cc68353ddfe
336be64a567782c156d2f5108f75eb9a
a64892203b4815c0cfb580039654b5d6
732441e86c71138c9e8d4a80ae032ffd
968a41f527a3e1ba78425a3cd0ebf3c5
386d4214363e7cc93f6a505c1b3d8f03
fb2f5a586962985ec85c2bc72776173a
57a6ce4f4f220ad499115234ca18eb7f
06ac291a795fbd366f265a6642a021fa
7578a07823c8c61f369dfa0c3338a456
4c5e2bf68269e9c3d32b04089e84a9c0
f7194a654623245bd83c4d20fa362c8e
83e999399f35737290ceadff7f6405e6
b3e793af83266dd5baab9ffb6da17792
dd3c8e25f69aabaaf68aea61a829a0eb
027a705f56abd91757e844efcf3d9f1f
e2a0bac7defdd8623dbd9d94fea98196
494d8fcc2206f3583dc04b70873e929b
b8d40b57982e678fa0f2680dbc8a2bb2
19b43cb1973436706b29e472236b40cd
25364f01c569f9d15f2045d355079240
6ec0e681b5adea1254fc77e531a38acc
d1894e44877d041668738c834f9f48f6
3d091c80fc7608019c20e35aed12e02a
04e3c28ad97436624378e5034117f1bd
c7a5cc4b25f32a64139157bb68cf8abf
a5e789486cb97aac0e2d375ca7305e9d
72e2718c04f03e45e735fec81a40ca59
eb38802b5e4a997db06d695e9edabdf2
af112469ea82dccfabe38ea2dfb5d17c
9f9dd51988fd3e1afc546a037cda90fd
6cf3de958ae1b3922cb4dc7f9827fedf
8252295795f1698da82fc05a76b5eca3
9ca64c914c2e8a939e389bc5dd57ddc8
18e5af72263ca95bd45946d05a604b8e
d6845cd7d2103780932f68641902df62
4c1cffe6172263ad17f439bc3df49170
766199ce2d747f005eab40b193aaae25
245eb04be4a6e76517609940e462ca30
12566537ddb13f97018443d18553bec9
515f9610bdf6ab689f54e5195865f9a8
d94ba4d550dd41b58c59c25d480e03b9
23e004c748d3940e524e733ead8d1ba3
9aa801ff9b8ce2241e996891922b700d
ee4d6229c0eb7f89e0adb9ad20b4c38c
7726750658fa2453c9fbe455859655cc
a782b1fd71fe5f9922154533f5d2bf8c
b3718bedbf3b472ab5893e78a014db98
bfae04b43d2bd81d737f1874e361d972
1c7d93152565e4d07ba34b35c4165149
936b7f8b0683b6ae6417598d4fca7c3d
3d8e7b26027a7fab77fab1f63f63fa37
ecefa54c7f798106950bab6ff9e3cbf7
0b6387f8576668828ba43bdc3ca5b965
d01e9596ffe37f3826bf60481ccf5fec
db0fe7504769726ac50d2629e4c8689c
e37d6d580cdbd7444692b63f24150ca8
9e1e0400f9229df3b31950adfb1e3112
8689697b7f11216e4f7fa2aa70a68f30
31e7542cbd9237e7769b96f6bb897758
e18bf5d5e22b8ae70b8532590e0f27a3
63340c2831aa481beb43a2ad655d1cea
4bb9968a47227ca186cc5ca5dc4e51d8
e44eb89836fd41c7e3bb1a1b113f1807
11d6e1325a9c6f0abf9eb747723434ba
d101133fd05b66727b043f9517f07dcb
1500dde42952639195bc22b492565877
ec88bbba342c178264250d92d9d87043
5a81572fd0c8d0c563d87e7da7e551fb
4cdb3a277405d89c86b5bcdfcc1e4cdb
9fe394683fbe3b4cc9811398d6d34fc4
e3c68b7d8572dd489353b41c0071d3d1
9303c3ffa9967cb49ae8e43d42148732
4f99b7a37e73de2ed24f631112212069
faa9c07e0317bfceb6c121aa4c18d952
713be5d6dedcaee76eccc0625af9a08f
c9f41adb193a8b93cd74097efef52d7e
262c1deb13b6c126362067d670fe9019
efb8a9dfcc8c40dc4569a3687fd77a06
adde9138a944aa37915becff381bdf38
0ffc7f626369ec6079fe299725f0f793
b37dfc5bb4fb57aaaba76f9f6e6ce70a
48433675043819f5841a6c38334d4b11
e7032cd2222b7d4bbf65ef63dda3bec0
01337caa9573c99d59d3b300179ecf8d
a4497c1abcdbce4d7785dc4438e5e7dc
da9f530d72d7011192bd5e4e3482f682
64a59fd423c873a770b6f75bdbbf472e
e51f650cd834295e8e3020ee387da622
2088a45bc59826969178eefae79358df
61f383381f9f09829238dec43e371ab6
1278698d03880cb50d8ecbef7f944e79
abdd899f52c2f91c6e437abbd4ecb924
194eecffbca78b8326f82d6f60428b71
4bb37c7ba8d7eb02a18a26a26e7cc50b
60615a1bf9eb601f5b4b6a81f85e77f4
33284bf4873da7b2ced33f4b4ed1a567
d952a3b0f42c549d5c3f06e112911552
145da7196687fd8255cb547cb1ea2928
d852051c059969944829bf53b8cb7fc5
8e00ffd2c7329a7b53084e9858ee28b7
993f3fa000a65b0b239c87effde4563c
993492b2a20e8cd03a26c50084c0179d
67e27e5766b7eda9818a6238ced3bc73
0a1e24b01ddd579c8f215bc6528fdf09
6f6d648dd10f3e927a2f6c4fafe079cc
ff72557003d0ed5bdad3eb5b0b4064f0
3fea9d0c2d4b42ee7be7cf512ea7d3b8
90865a871271bba051ae48155dd4a3f1
45a7e6211ffdd3d5efaf7048c10a0459
c03c9acd147a0955efbc81fd77d7c9c7
330fa3da00cfe71a73b4d5c4c3e9ca6c
4e8d9b6694c13fae3fe8dcc7ff0c6b27
cd968e06e76fd824c7c3718a7245eccb
b127f0185975eec6092985a50cc3c4ef
ed5622dc9b1a3bc6ac00fe8744d3abe1
b543444d74bd09e77d2fae4097aa3d6c
58ba876de454f5418394280ffd39a4d7
c5ff07d7a80233765dc6a204597ba378
7267ac1e5b593f711f8ad204190f869b
00197618ac11122ecffd3425d42a3f1e
f29337ae8196d28d144d406ab2301c0e
cd6137bfda24da4a9500748b2ae53a7b
2637326e72a0661a4cf8a058ecaf659c
3abbea5efcdd761a94bc10729b608dc2
b045f21b0f06d77699d34ce03a6a3ceb
c660f07a46ab0a43442e45f781c92d61
d508c80d4cff7b78f206381fde8bad57
bdd1307f016d5ec806f63efe1c2372d7
dbc3538e28232481a7b963513da35426
c05d8b702cab1898ef915712313c29d5
657c1f99a177bf602e742d8a4a09c77f
77aae00b6eee9e25d61b70002203794f
15e8a29ffbc317f5583a25ebed0af6b0
8cad31855114c3839bf5d187aa7e1d2b
9a287172e2e149ff8d0daa120ba26767
b48d85a7daa804c65a0f1d1554183e37
d7dc869a340f065cc201031a7502ee0a
96477d59d5ee55c759cba7656fc9960d
c23d4e1394176b87e604e07beef49cc7
146933c72dd6defd4c7b3e4e03619ec2
2ffa2f60c0cf1b85eeacdefb6c0b515b
0f7b4c9d8fbcc969ec63485b7c595323
d92e5d68b76ee886ebc68a946cb6f952
097412ced6ccca2af453f1779cfbdaf2
5c0a8adcf7c63f2d1b566f4c733611a7
65f66371ed016dfb8f0da7ad2962cef1
914f99d5cc3134429650df04f8b0bad8
c9347d035464a0256ff34947055a001f
3c7e72d69537e049bf7e14e53fa6379a
043e364666d4ec4d09472d3bb7bb5837
099aac910fb01584ea168dcd6381a4b8
87cd20f88918dad6b915a6611b2c8259
ce14c5a8870ca37f99163012984e0804
efa072421dee069d6bb502846c936603
b51f412f792c7da3a640a8328189e11b
8fcac73fc5535ba3fce54907e53e5571
7027bb10521c815faf779306a34816ed
97cc80de6d7ec415679e92545a8d1635
d37b5c4bf3a56d4ef9d7821cab50beef
80fbed63f7628017bdfa2d6fdc5931dd
f4011b40b2491ad0620dd9f2941a9ae1
bfd9bf81da5ac19a12cef92397620b7d
beca92f456b4143bca922956cfa4602d
69f58967e7fb9079bea51ef0c30af34f
06c3cb3837b2b9969d7a5a7f6459e40b
bfe21d8edba001b361ed8d9b744e4d33
b489a1ae4268a457a0db401342517d19
a79c122b2df67d004b697654223d673e
7def1f494b1d2b834f4e83f1a7e24432
dc1350d5ad3eeddb48d7db645155c58c
b2b5aed69ec86938910db8cdcd4ec56d
1bce286ce2821fc653105e580dab2fe8
1033cea86853f67597f720d69816eecf
b6064d67595e689b995ba04a86758389
5b34f5ad799f85fad751777c91006d40
d18b1ea37b66307a84092736d02642b0
f4529579778fbe46573bcf86e5af528f
c18b7dbbcc43b38d9ec4a6d7ab20f761
797d0c6b75550be8170061a02cbd86a7
0e8285bc7fdc89f17cc33feaa2f23afc
879252f4ee1065158d884cda313b4668
5bee5e2771dbe8e32bc55d938cc7cdb1
53a03bae698333637f278ee42ed91997
c6cd67bdcdb682b260cdbe701671f592
79b403cb2f35d295a47d7a45a8fa2ee5
ea4295f00312a3f4b2f7930384e692f7
d801fd5a2fb36827c6c2c74acc473216
690715819dd01d16d37e01f2ce1bf30e
90d2f9332fc41e3fb4ebc6aa95f8645e
d0fb1ce78210bee0555d438225a8af70
623a0017d61c0d722a99cfd03c521410
fabe1d80263acfbb99a501de7e480344
0a8c4a2712b8ca4e6adc2c523bd4e0d1
4959ddd1b11d3897ca94a59e362497e6
e23c958a474b7200f9f37fa06a71add1
42eec234b09df9fb1158515c7cdfc1f2
823ce9eb8f44e5dedcb63132d2f14964
124c47ff43530d15182b8caa12344fe4
7134d7199049aef6dcf01757db377872
030b346c59aa679f4f1ccc7d69389012
81d0425a964277c0dbe80b890fbdf976
e872e3c603fb3c6af54e85b1a25ec461
34a8465e8dead5622607af91adb19042
960e96ede2b7cb3b534fae81dc94a7b6
6518c7ac9f141ab5c60baeb977bed189
038e9ee82d6743a6eb623e368f76fe46
5ac3022cc36f83ee08bd7cc57bb7d72d
dc961cff81c313fac1ffe9cd35f32617
566941cb772840c2e699ca7f3fd64247
df221844e262364b567791c8af3bb720
3af10bb04f689eb75229f7e6a72d91f3
adf7fe575203305c4ef2273252e93a0e
545451779dfc5829898c01aebdd228d8
acf6234969f43aa858252e8180362b96
9a6b97b0e6448493470db36ef40ca08b
f4727565706bca9c24de9ee733bc49a2
0324c7ade471cf9cc466af2d2b06b88d
319fb023fc6ffe4d6a5bc1f558a1d9bf
8a66cc31b767873b40c5204ddbe79aa2
0ff46b0cb48ccd5706f9cc5257100664
24accdebc66f8d5da93985ba730a13e1
d9480501ad9bd62b7169d9fa17a0f805
52ea53dfee9333e33fd30c5e5360fa76
1d0f2b469277050364a589d0a6b030ea
7367bf6baac994f82ac3d565d27ff0b4
562e030f42bb92b4cec86b21e3cee57a
4c38960187629aab6279b7f098ee6e82
5096b91d0108df2577debf4d02332087
a0c6e1d2b1482ed794ce100c1988ebd0
ae55f453ab7566f4f041154c20b3b09d
ab23dc9c28a8b32ec44a52cb51a83bfc
56f2061c7ce194cd9c774b8dd9728ca0
f15ccb18370ec669abfe8dc053d51c91
f12e9b3bcf5614275dda7c1baf55a055
c0428b485bbdbb52f08cdc7545571d28
1ded3e250b57a8ede190fea2ad072ad2
29fed6eb5835c8ec305e3000101d4df3
79097811a6e53241639e06499de103c8
21ea64f6b1e922bde8ecec06d39911d5
120457de69d10d6e321eec57e2ad8bbc
1ae9c97184eb6fd517b1a1c0a8843ac2
ad30575d5f8c6309c50238b0ad07d366
a9e97bd055cff2a086160de5e50e2a93
bd4e148788bbc03c6d7b7ab8119ceee8
cb8d80b931af1be3fed905618b15b8e2
196c9498c7fec89438b328e843dd9f09
c7e028098a5e9122129a1dacde75ee0b
63592441c54be5f93ed074ca4be21e49
30bbfb277353e210f53e07e88fed5a01
1e09049e9d3418cb0e383fb713d2e7aa
135a3a67e549031e456a51361de8c35e
60d163c6018186417885766809a3be7e
14432d5260c255abaf8c72c74a3cf535
91f21cd6f5ac062c890faa15f088fab1
8596d3035bf324f440cdec758b5b32da
4b8edf24a6a7e38a6815d1e0c5abeadf
5c6cfcbe0ee3c71948aacaa54e39ae7f
643eea0499fa643ea36ffc08ac228462
f8d2fcfbb76ffd727d456d36bbfb43e5
3511adf8ff174bcda6ac9197a49eaa2f
6de143f83038c52d1b45afc39dd5e1e4
45dad5d62c382413301ab211e90b6708
ab309166c6db257cc22f6376dd3181e5
c34738b346d68e66b946211dfec57b0d
45dad6749dffdb42ddce344da13c4312
d250e626120224f20af7fb0a6500bab3
6cdb79768a3341957c41525f09b97f0b
92aeda5510728284a6692025bd7d19d5
f7fa28a725a8ae20e89bf81d46965c90
2f3f4d49073d96487ab7aa38c6dbdcc4
c972fd50f6eb8643363e6a18de55053f
e00f7790c1a64dc5e55cd5e31eeb70a7
bdbe7f64049ae716281b4d4cd737cc6d
4ce3aab88e66ead314349553a0914951
490874908a69c4f50ff49ccdc5d936d0
c4add271be8383464571e161588aaa3f
9784a540b047760f753bd170ae1ec01d
8477aa69d014e39ba10f80e6441d6d96
233348a86d2ec03defb81e53bb6b5f73
e2efef7bd8a6fdf90b6eb4ab5a6402c8
7d5245c9c1c9fa05171733f3056b2a69
6e2c1f21d10702f00d713e5d74176125
568c7eb03e56d792dd3b097945a2a254
78f130ca1b7f4e168f1a79366fdcaf31
4663533a133020cf540dd64586944ec2
0af6ec6c946b2c8ea0049043ead2daf0
6d6feb39d000b1b2af990a7285555530
0ea846b0436e087c9601336ae447ec84
d6404240edd84c39755bda29db36538e
15c69ce503dfd2007814df2dcd641337
f57d6d6980d41021cdb669a8b6a32f56
27d06b057aea7f35d11a4fc34cb54da9
a0f27875e9123e09be7d6ce15ceed6d6
43d9a38ecc79266a9d0ba9fdf3d239c5
3d0728268a1666ae97a74ef026178098
ad60e221ea35442b84d4fd7e93a20ee8
1f88ad19babcd269cd961ecbd26e6a44
16f24b121a3c6ff84406d8c06ea6aae3
1b9cb19e4d087cff567ed006850f4d4d
4fe6ec92af12cab92074d34d59ac6077
a5623867f342de65f1ce20a2b5ed181f
a5c87473a62ca8966ddefd079ff5cddb
391560026aa64526cc8d6acbc322db86
c9f501aa9c0cbbf9d171dd325b52b1b4
7d68c63cf49540d19fe7a0531c0890e6
55e9933d1d542de01c1847cb2e3b60b3
2465f871cb0360c709509d672d280613
dc6258a26e2be539e38ac000bf4204ee
06aadce308603bf901ad03a6640cca86
c0db2ef62dabf7335034af4acf4df9ff
b84f0a01182ca2ef26b0b4db13dde31c
2e94700da12040c98f2b4726b6bdd04e
1218b6a5619e6815dc3d4f6ccd4fed84
b2a3bcc14419c6ecd36c42ec721255d0
91e8e16ebf9852a883a7a27f84dd3ea4
70f9596f3e9b730900aa1133f6a602cb
32875e1e4e1ada4e1050a08a81e142bb
4c486c99502e35a55650756492de7040
a7fa9049c40bdbb89c7ea72e55ebe5df
10ed6b7ad183d10c702907b6305a4487
44b2840ca82c1b55786f3c8229bd09b2
ac7130111e02e22dd5e37b52aeef455a
1aa2c384e79a97944c36ea9adcddd7db
f41adae2bb494d9ef8e7a6bd8d66d1a0
e328701e4e99d78850838db76b389177
c8ef4551468e33d61039a8b789fb4762
a82689d072a93cf96c60b2ec5c4ee83c
74cfd24575e7e5b53181e5ec0ad5e750
025ce0e8a409bb3ff114fd796dd432de
6c718b28bd19b7c508b577c9c4cf4b41
026304a3a56000688905e5b80ba3da87
268fbd3af5ede72170aaf9e47e3c9106
8bf780312ec38927f59b3f4071bce2b9
5741dcf6ebdca1e6aa4005ee090d6426
cc7aa9c22b3d38e3b8813078e8fe03c1
1a809268eb75b4a16b61bda60978c0fa
6ca5c61d7edba5bd2d0303b1dba25a70
8de9e1557cd2ea93ac887744a2bb8b85
adc7de73accc2e2fb7d43b4e8da10242
9802a1e7cc5cc114c2ced2ed96090790
449ce0732e1c96818938506fe1269a67
926948f31ed0040115fee33e2769832b
c6015c6a6dbc8f0ba10289e2b5c98000
50a73e64c01f5134ea72fc438b971813
28ff8ece5245640ea420477bf026c071
045bc016045e776fd4611851ec742880
e8c934e8f67f77c6ef979d33e4645a2f
06ccb2b41de5f0d5799683d355a4abe2
1128b02bedd170aad12b990c772ed12f
d721dda3c8a7be40c5c63f6bcfde794d
11770af1c85637da8a4e2a528c807484
214a2c66aedd5e63a93c4f64b839ae28
b6dced2b84e1b172e5b644bcbf952a6b
f94a6ff757e82dc8c556c1caa0db747a
8dc60235dfbad2d71ff45cddd9bdcaa2
59b24bc2a4820b4eead54fbe7059d35e
fbfe8d784dd7428d7fca93a244b7475f
1ac303ba8901f01d68c4aedffeeab2a6
6a8533fd4a0a36ebfa36527730f9e5dd
cfb0ddc3407cd474465cc01ff748fdbb
d62819da340f94ca4c71d425dfeaf3a4
419c07fc46737d6117496d1562dbbf38
1c5e3574cec0a9e4f2d359d93a0ac889
3cc91b332d155079ae95a624feee9484
f298443462a06844d2b6570c01fb5eb4
80ffa2171d343b50169be4386c28b5b1
bc540fe50698d9011b626ec992f64447
577c77e979816be0a3cac1078942f329
6964711e3cac993237167f37122bc5cd
3652d6c1ac0faf40bd1b1edab5f20286
4334116f46a393103fd52b066f873d29
e9668de1a83988e7747e225dda7ead0f
92bc31d54337dc379cf229b00116da3f
08e02050b0ef6bbd89327f084b70b5a0
89511ef491c708255f9cd69f9cc72fe3
f361390f5982439ee8c976a264886e5e
9e3ac849b52ef521139b1663bd13c0fd
ca6cf7d22920389b1df65a95a18c888e
ab3f3500045afe44a7dbd765b5a7c5eb
bf685673180acb82027e358e47a6b5ec
477a283f3a951026801ebcca554bad68
ce67a91c58dd7a14b7d1545d6942a711
317144aa3750e62248d1c417e942a99d
bc95f96279df350c0035bfb36c687ff1
ad8e6f84dbf39e83e9a95c3aab93d188
57957782d4e50fa4fda0275779ccfa78
4f4bbe4e1736c133a0b68a8aa587ec39
79243dbeb17ea90bfde9f6addce475c1
8d42be2855867fe0d04a7e0994d9bea8
cb8cb707f788dbf4d96ffd0fcbf65992
fa08fa8766eed234238cbc4413dad089
1465885c74d4d509c464c165b41eb143
fda2402647c8a8d8a19194faf3f205fd
42819b1e25352e2db5201aacc23bea7d
ce995e7fc25ef9da3c6f61fbe83164ca
22863366ff7a013af38d21720f7ee3e4
9974c4c72ab02aec5e7a1b4a001b7680
797957824e648be234805f4d698fb070
5936fa4590263013024daa83e67b88b4
f843de41138d9ef960575b8793bfa25b
c62587036adc5f7dc2d776699e7a766a
49457d4513f8131e2f9588918c27be39
ec647a3881f2752842e7a731b55670f1
93e134fdc8bba47d9a367c2a61e453ea
b76aed6c7262a363f0ab59b782be33b3
f3eaa9b43b9ee7c93d908c9f16947367
bcaec7757568e30e48d0cf92b8f6e5dc
ed4563f9ea2bf50a48b6f7412a96db84
343c08c9a76019f47eadda22d8591857
0d2efa833910f194aeb11ba17441341a
94b6b7968b32ab4f8333eaad74642d3e
95469a809d57f6dff843c8375d613f06
8a8c5d78c051de60809a04174f984f31
dd94569d8ece3bc69d7be9426167280b
0caa179e836de5f97dfea8b22657aa27
51b10fec5283bb064cee665b09d3192d
995d921f03c2764786428c891a31d7bd
1d9302c4bf478fa35d89cbbfeb2b8568
fe8a4b3e0ece9a8e262a62acc75ccf29
068ca716a667ee9804241092dd5138fb
c32f6be05ef551a4e0a27a9e586fb3ba
4f5ef7840cf1468558fc1bd522ab253d
93f6b28b6fe0bf013bc7518509d7edf1
0f21fe91b6c0f127d377ae879ffcdf18
944f79fdcb021205e9a4a8db132da2f6
4663097e7efa0a10b8b93a4b903059b1
b993eae4c1b51c3cea912e9687acdfa0
b2f812ae3b6af44a8f3e1163ec4d69a1
7756a335ccc7aa285b212176296782a3
976f7c8e3ee573bb851d874cdf9f8ce1
69de9c6ac95b9a75e8f65de1acfc5812
9cecf82109355db1528a0ea8bd1cf529
b2ca7d8b5e12f5e045ed0b1e5db94a80
7526f07bcc44849ce9d29c57d982bad4
bca4d4a511dc44a0775b09bfa9c0032f
73896d017dbb60cb85db4319599036c4
3d99793118ba2ff17af824a65377655a
dc7d098ec4d8553c9b06d9707f349b3a
f49aa5424eb332be59ba1ed2545d1418
bf82b317145a48b2e8656905f428dcfe
014aa17249ec0162a4f4fde8c8584e24
263b1d3b391787ec268e790cba44f910
03a7bde866ee4c696ed0720bc5f06d67
266195d7d97813c561f2a2b45522a8f7
1f2e0f1b5ac89efeaab6ae54772dea20
6d8265cb0f76605ef9742a8248e20f5f
af4de78d43b7d33144713fcd482a02b8
0c3a2b29a090de24b3216384be139865
06efa8587c97c26bb2f7769e3e8aa9e9
e88f86f3cb7c05506f407396d8c2290c
72f06dd6cfceac789c42a72e4887957d
df0a3ce13b458a93f8979dc5526b68ed
187dd65c3798ad505834565e35549b9f
57a6ec7b49dd3632602010ed85977b85
00849aa191c7226ba04a982c220a87b3
007207e13c0f2d21a56948fbd10b122c
6377c084d9f0dae706a0d38836bad05a
4a435a5eabfe53751f9280cfb4777c11
55dd9c0eb5aa170c737ebbd8fa28da07
3163c8faed9c439116522622e9c66530
908ace21cee82fcd1874e90dff9e1bba
a34903d5aaf8257321f14bb7ddb0bb7b
e2fc21a5de6cc582f34a29eb4b20cff8
4316a95caf9c347c6de3ee1b5ad750fd
4e7523a0be8773f58ce2a2e47f425335
0e77df5fe3dc1c9013072f65991c8395
43c29e0b883ab72a225a55eb89afd95d
b1357b896dcdd31cfce9a0ff2046f7ed
af4ab31907f19407ec6336b140edd752
ab9e7deb5db6df18299b37d1499159fa
b7c1b50817b46b461cd27c8d7f4feac4
a11b16b54bb2036741dc97ef8c1ee8a6
b36a4718ad709467918b251030a7d322
66d99f6edeaf4f3ad5a1c247173f1f98
84cc8e462e0c25bc18102d0e74dc5630
b63e96a3a89dbac678968d5a4bfd9855
880e43fd7e99f1a59941f1367317e4db
5c6058dfc6abb5ba4e0460a20c42a09f
e235d143bd7198ff30a92241da16a381
5ffcce4e8078b6c9c21a6f79441e37d2
9a37dc791dc2a50897c27b4aed5b85ac
80e62f659e0cbc49eca67fed577a3775
1aa5ecdfb6278324f0610d004a8a71b1
7f3e412e31337025e9c616a2408d3b3b
25ec302061fb2bfdc96d56234f45176d
c3962d74cc9cd7441bb4dd917b8d8e69
8c819c8c40e98d489a83b47aeff8fa0a
126a3a7bdf83642cae74e15ae0a8060e
c0e23a13057b3826b3a3704b335ea4f2
cd7bf2f4793162843600b37c8211a97a
d85fdd00b2038e0d3182f63f953c21f4
7846ab83307486b7d3ca1e2fc586a7d0
42fc488de3889631ec998264157df97e
4adef19e5715ba96a6447fc3d6fc3b08
431fb8af4349955655870fffe3a1e0bd
79b3cb661052ed01431cb8ed88dcdd07
8f4a8be0b26098f6cef23902f379e1e3
373a11fcb52f7a6dbca0bb5df9212958
c6ba30f279df9dfd2b39b92594b5263a
a68e9ea9d857696ef674d9a73bab2102
1775186aed5510b60372d5e3a14fb372
150ba22f674d866da5dee819cdfce968
128702482b6352f08072a4981446387f
586bc9f5535126b2aafcf6d987a8c15d
804d6d5a35edbb399bd4237eec4d2e1f
f5b680e6e6d969e5f2144c7f3213e465
f9472f275ea34db609ae060aa633d73d
5514bc0e19cf84236a26502595da03df
3e0a869d275caeeb6bebe7e621c42d70
0948aa333c714963c680f7f41190682d
fef4797504591e5014a136c115235ed9
f32086434c076f49aec1a814a8f6fe4c
0caddcbfe76881594074c5169caf96f6
165c3915ea2c5b98c0cf2c6d0a4906fc
3546abda8fd11cf280abfc5a25a13936
783eed1e9d5488730ace630d5a992207
9096dbe975dd0ca2d463fdd317a748d7
38578cda6c95f70bf063faa35056c11f
99622c7a871c580f14e3f1fabea4d93a
ea38d5eabd1f1b7dc32a5896bf128ea2
6d622f34e84fb2265274657f24ac05e2
d9343a02a6b547b1e4b10a2bb87ed086
a3308f5f655b702bdf3e628809af3244
10ed2ac0ca7147844f516cf9129fde73
e87f800fac42faf65ace045cfe4f27f1
3134f90bfa4b6a85a8c497d08a23e6b5
6deb0598e927b93cf4f9d43edf787872
44e390715af328d122ccc235e77eef08
c9410b460fb2d5c492d43361f5ea1912
526e7072d5694319168391b7dc4e57d3
a463b5bfa217d79e13a48b7996dd85fa
e2a8867e179c1f64b7821b342d8eea7c
5e31a0da51934129fff05c38c44d3274
4ad58a87ea7789e318f3bc3131d6d0cc
4140cfb8770a471654924fab54c3ab79
24df7bd70f4e470fbbe33d50759bcf4c
dc87c20f548d45000dfac813026b4f57
e68eb80fa05b110dc112e7fcddf8f666
13d1a997de2d73bc268f066c35382255
9d7e34b71709a8d1f420dc1bd4637645
5db8a1ba79bb82b755674fd4a0b2596a
3b8f7ec7c74160f1d004ee6a4e101361
4bb42eb8e2688d25d233abd3e23941f9
b8853737f588c199f6f09f83fbc2b67e
ded3723ef6907eed35f11f24de2f8832
7a4c651860550dd17e9732030726c13e
f6faff01c2bf2b5095aac0c0d47143d5
ec30be5bb5b8e99cedfed188f6515f06
9127ee02953030dc3ff4074dda310a76
92a3099233b207f4e743fb04c3b3afa6
1901ab8edd99144f7cd57a420321b076
52b8491bb0a1ae464051680e254be3ba
58061b9a279498c6b71b51ab5a115db3
077faafde920ecc37944677d4e8a34bd
6cbf50db8b79072ca13daf2d2cf7cc80
69d313cd5a64806f0419556d832febc7
07d86a1d7e37fa9222e7e29ae27272d2
186aec4469e339bc9df86834580432a8
9e2b98f6267ff121068e64ba65b4b625
63ffcc9652845f8c822bfc5150b480dd
50861f86a9ab11955fd7b3a9c50b4afd
02f587152bf9d59cee9f984ed3f6e1d1
776c4a2a3fd254f656d4c7e67b3e2ded
7a70ac3d29b77d855801801e8553c4b4
c3d9c485c206b52fa9b965452919aeb6
57c7424a2930b41c94dd58c9a37aa3ad
725828ccb541b053fc5ad0d05473a04c
756ba2cef8520b0e172f4e393fe1bbce
580c16b70cecfe737b2a67f1663078c6
7bce2c93030f4e81f25f27051027d21e
e1ce2395aa4ffbf2fc29bafd91701457
e86285ad41dddb322602387631883af5
14eb54612f94327deb7f3e6c39373f4e
7e4b742d18209465eb9a7b9a7161acac
fa2a826c78c5bf999ffb817a8af312d4
8264721eaa16bd5e5dd7275a93034c3d
79e053bc6c2cf989651f2ee5e6a0dae9
197a443800b9701b15b6b27599fe4a91
2b23aa89f80ed6af0da6b151e34ec051
f0818b1bce98b1c30a4b684a97a12799
82fde9fe12b4e5ed85a5dcd992a93771
c239f0e2cc1208fa1964bd5fdef2c7ca
c7b4d8431849e8a255cc56dfcce0f894
0cfe7f795e79ca0136ec33b739e42243
7a877568c29001aec286fe77f9352519
935486a582ef4099fef8d1a71d0768a5
d21d94818cfc2f768821bac7eabc9428
42fc7af2a4d5f09288c0bfe2f18e92b4
c59a3b53258c0b75b722f87afd88aa99
54993557188d12c040111b3d18ceed36
78f6b26f4be886dea574fb40ee961b3b
69094f2c6b2fff529dce660198e4a374
1e82c7428e67e712768f02c50db7b551
71cc7bc69788048325fb311fb80a16d3
a3c44be6a6a675ef8f7d6a83b3ae3d47
b6a8c6bd648dfecae6eeaa4b6fe95ddf
f95460edc872853b65868be0c7d748b4
720c9f3842ee94851e85b1b223c164af
a30af077ac40a06555e2ee8f0eae8b9b
6ab926c9be4670c9556b6a7514b45714
6629a56274fa5407b187068bfc92d6df
8fd38ed9d38c7ad9dfc9f48db9024569
801f468cded1a3457621930f34115871
a1f97526f6685dc741cd9e321db1ecd3
4ecef780cbd30a0c23fd0244180dfb26
34c26862fcd2f4ce9121f219c3032052
10b66264d20e5b183364ce50e96d5961
c8094c0a460fc30e4325c9ec11d2d154
5e6eab2d7e77c6cf4b1933f1ae4ce585
fa48ebc2ac4042dab550425843468bd8
8ff0cebd403ccd6d86b4038582317e57
164816a54d4748fb5a0c85a04d35813f
467ba3500be80b60bafb91c2300a29d6
52d69593b7aaa4bbed033550f4c7bfbb
4a5df7def7518a16be594e6325b1cc4e
824b185befb587e5d7c3c64f41f23925
53ee4b272b3d45d37b024acfd71c883a
f7563dfd0665b66a2b6224d56d26c80d
269695626a46b48d33720cdd852532c6
e4359821639caafcf3484a3c189695f4
f15214d7c70befda7a829f9919c4d3e2
324bf9ecc8e65bc63db087b5b2fe7278
085da40ca0177c2eeaff5736b6fd00a2
335bf0f2f0f1940d50349fb4e32deb09
e467b4531f6e9ea1a2f6fff430da2681
c081f35de4827bbcf1f0eb88f56cfa03
5a4e605d7694475d800e841588a849de
bc5b6463f2a41e4e040699a8d469614a
bba1dc7b7165059ea3b96c0960781d5e
7d97805a35d463ca6d5a6c232293c696
78b64a4de8b6438333fc319f3294e3a2
ba2133dfa0aadc0c3f9f29da07f92674
dec464ce8333664555081baa8fa6108b
6ed828d5f0c7b46da777fd5c0e1a3825
139a02768fa1b8e1180c13310f6b1dcc
2536a5fc991a6fa6b9c437a5cd58744c
a65830b63b838c38c2f54d414a853dad
4143ce1889d61b331101259c7ace671b
26b4fdbf1903b471e181cfc928c3f853
7828d948ca6bba4247eb76755eaf07e5
f09e51f3b04964ce87b07a6e2884f767
700d40dccf7f70b9d253e23ea36e735b
22c0bc33326793d6e6f0902d2b0b5126
db6b3a95a9777233fb95d4a83c655cd8
2a4022ee7810998217e47741396d6def
8185672a952c404e8e155665c0bc009a
297041ed2d8290e0031768b038455f41
32cc4f6874b9648fc1c8a0175eb2a256
25dc0b50fc50d7819f9012683de0a2b7
89bc69368878f0c687870a610fc6f7a9
9aa972e9df32816b1e37b564888899dc
38593c6a5047cc9872a345294b6d3073
8b5b79e530f1b339ecc3f087ef34543a
750d58692c9fe098da9d0a28a7d041f3
db8fb346147d1fd266424ec7d293b93f
ed9a641e343cef3eb10f22bf2de515bc
6b86725e762f09c187e7cb908b897aa7
7c2fa72bf0bb2810b5d9d91f7cb63fc4
408900860728cf143939454b09465a9f
e36185edf1530255176f03d9db9f1d13
164e965557bdd33989b7b363b6af1db8
169ba803341c51fdec1b30bf7b21fe09
d54c73db8d715a800a4aae9361c940a2
b0631a8d147b5f272af184b3f84c0620
a94fc9b02a4d2b90bec07fbe8ec7168b
13d4c2099fef2794d451205f884c7e2d
70cc8134de566b2681d6b2cbb95e819f
dd5a0244d4bbaa4ab649b562619784d4
fdb31527fd2ad9d665a99d1b46f50678
44f681507e395d7fb4e0798ede51e65f
7415582724a846fed87751456ee0bc61
b97d7624f34afa2103fcec8ff63e73e1
7203e5b96245723cc210ca07ce178bb4
2c0f4e081b7f8209ce5dcb9b5f3c507a
8a922b92663f36c6bdc14d42674cbe37
4c77c037ad1c58d741350b939a77cf33
23ebef5324550f175698b0458f5cc1c2
0ec45637bce9a031ca8ac17103f876c4
63c819424e24e0fcc6622ff76c2a1c43
013d93886c57cb87a07465da7149d3da
c8542f82eacb3645e1eb61b2e1a3af53
4cc0aacd9a337c1c88dc782316393876
42f6539b352fdbb0cd538a1ef753f7b0
e05fbf741d72c8a32999daf2ec6cac66
3893787099605f145dae30909d8301a4
dc5b0164d072c43fcad67b54d82bd4e7
9b758839f4d89e1d621ec41b7de43b34
0118867937ad55f3829998e08824f216
2eb9f512def07ef61ca0dcd0457a44cc
89e62e5ca7bda65b6f4737ebc970313e
4c5d6e468f618cb18bf66611eeca1549
9abd8d3dad8d62d015f72d0c5e9d3b83
30068495aa71e8c06d41075bbd655400
115092aa00ab452d48c7647bbc268252
9524f647ac02fdacb6d2c74a05e9d02c
71bc271eb80b204fd12b4a244eaa420e
c7a8593a6ffd28b05666429b5b1510f7
c84b1119d4eb40936b4e944d458f1c07
d03f26ed32f2b79e3864aaa80ae149ac
f11ccbd66694893dcccfdbd1389f5d69
4c778770fb418c9fde5c0ae8d5c2e29b
ddb78573b0bf0e94b2de2b6a35fbaa30
c6ebfe9e36e2e2848c81df70827c976b
0343c56677ceb8d05237cad043641e0c
bcd817e921ecbef1366a51017d68016d
fe4c73a5ece2b1c22f6765352524ebd9
c73efcc7be4e28b942f695a4d7b4f39a
88e07fb6d60bf92a0e3484d1e09e5477
9de1fb03a65cefac9b345993eec53dad
e1921d7d8c41d097f94cec46bd576750
866498d17a5d4c254cc5e27b74d1708d
2e37924b8620da4d30ccd804ace0d5a4
d7a91b10408e6c67928c945a79c44846
34195241e5ab4a97789e4c55809c5104
88c651f97f4d0befdf5dc5f3d6081e4c
60a6c9e27a8eced62eca317969705f9e
9ad27fa2678849392486aa1b7b34143f
82a8a6eadb3033ce2db7f878d7ed5ef0
759af1fb2f7a2603148442cbcd34a8c0
ae9539dcf0780850e2aa3cbac7ffac45
88165e120a7c1d85532016f4eba33f70
ad9fe1dbbd7d80eb67da7fcc4f240147
c5ba392b29f01483bcc2cc2731eff529
a4924390d4fb67ab2ed88a8b79267244
d7f79e8cc8b9477b94fca7b1364d410c
bf5c25c91ef6652500b08bd6b8c8b30c
68c315433294a72c349f5991c3cde614
044cef78ee835d3091395f832cd66d68
f6d443c1d0e4144c9ed24e29a3ab8bce
decd096bff10d898503e759b4bdb7503
be84f9b5b8c1dd3d0bab8b929d35bfcb
e9c0111f062feb6321c4c3e00e8d0bc0
541b8261c13695a82fbd849d8acf5ab6
a86346995861f9bd7cb0fd241a24c917
627b332ee2b6bf04e0cbea0344012569
a19c9761c90724ef47dbbdfef90a76c0
84a91bff209fb3943283124bdf585ca1
1176758f30f630bc2f5d4289054c0df8
f3cd63c836203b7dca3f842894bef814
575cab629ec3c2cdfac05c595e684917
9974ce4c720dd081ad94e670921f77b5
be0fb1cf6727f724023484fedc70bf58
c7cc5116156e07872ef073f0fe51d43a
98c5c443f96f0c113e4e4a43cf86f619
7808ad94340da7e021e34e87cf61a4f7
b521a9d81da4dc568d034075665e8b92
ad18a900541a25e253fde661106efbfe
b9f4736aa3bc1d595ff443da08bddeb2
666015fe064f82b3075f7e93aac6c2e8
31b99711241ee86e8bc699a0fab44da6
e0919af3c66ad368e4f81ada2ead7e8b
c6115b53dc454d681d885fd1ae34c651
5c1d3afcb9d3b2d6448c359339fe2a05
21e130f64ddcf3e62be64697932c97de
61ae0f682d6fcd46ab7647b9f88f670a
273fb68fb3603b12211877d22c731198
cc5d8197c7afecad798d135ca58e66ed
e80a64f80305cd3c2c45a7308e0d8a8c
7fc579220edd53f121c4b34522d66d49
71d4af200910dfa26176f4fca9e460e7
50f50a8b99745999f065e99f96bb9d9a
f57d633fec8e78456c9e07ec46552218
b29f289d688e178eed77c6bb4e29034e
90e429ba72ec9b2ebfd11d2714ea46ba
b518018577f4043c2e38637cba5c318c
b2dd8f57922c7679eee5063c175a6440
28ed16de9bd73835002256dbd7725748
b5c5714f36f96c6c41dd2b2cf21d2ae7
8a91476bfe0f7b104621fa5dfd305632
5ff2871cac729a0bd5dd12aab65423d4
33681f6dcd47df2f8863cf7d06631e26
9de5f564f5c9bfcec59bf429b3b4b5e3
b2b41890fefe7c77cab65c35a455a8fc
bf619f8ae06d9ae4019628874357cc21
0bb134d367200d89426001ecea8ad4ba
9f3eb8ee04ead554f47e1b29bb1aeade
5ba95d3b8d7a91d8ec732a9d2ca6e9f6
448f9c4df5fec78552a77b0f90df23b1
4ccbd523b5a2b4d5dd5094e2c0e1fc67
ba940a5c999501033d431c8a798f49b5
23c8293174faf22e506047f314934278
47f372c456ec6cf265b7eec583b0293a
b0d5dd35bf489a280310e4b1428d390a
b7511cabe14906b82e3904f30b000afa
35c8069f62d7993af0ed8f550dba1f0c
e507551ed480e64d97d57609d410531f
658a7acbc87a9681d9a756dde56d2bad
f900a91795302e306f30d3f1b3608f65
232fc1f03626ecc86867bc9c6545d649
ac5c85817e46a6813c87a542fb6a556a
2365af891a5915e916d2a78866906040
ae3d03f4645f546cf3fbd86f3263bc97
57d1add6de82341393340d5fc91a5827
553e9c43ea075815cf742ebdba7beccc
0fb0e2e1170492c925d5b184be183644
eb0dbf79c7b6563768dfd81a7ef77584
d3415c9c4faf662b22938528781946cc
f401c3fbe3677ec43a60ad66e45c769f
0ca498874ddfa1c1e4c07f001da33bad
bc58c141c7253fc5ea731a38e538b9bb
3b9fa76ff726a9d8ce6e2f6df1650134
b943a8fbbc4321e1a62001fd52d2c307
83dd59f5875fba3f1757e961161e3afc
6836e8cbb0486a3d50482f7ba7b658ba
bad87c0d072109da204fdb7d005baa9d
709a6c346d053355d281f16a8e5e9f3d
2128484108d65caef20b6bf2c3524036
c12e53b74c7a392dc33a3fb35b98ce48
e63d490e40a87fab6ddb19958c455b2f
937641f9ae2e59ba9e1836fe76910ab8
a8eb3185b6cf76e7015c69e6ace7e94e
42155617df95ceec039cfbfd7e3af37f
6ffa1fb7e4c174212aaa5f00e0d9bad9
fbe863418ad15ccd9382124fda08fd63
4671a6e7c81a5fc60bc64890fe24adde
783830dca68be3d4554a09abcbda7450
2bf4f88b0ba12cfaac2e717b3644d9b2
16fa22c5de046a1425e025b3af4a1911
e4b9d27b1ee786015dad777474c23cda
c1fd4ef59f4651f45c1ccb1d1900b7e6
41d6abbf28c55f17c9f3754b833ee5b4
bd5c1da45dbdd55475421b07ef7b534d
66220436b4e127f081c5facf61669192
df30260b5c259fa470db4a1f2bfda544
ed34022b90e90957bdee097af9177595
edf6d9406772208ee372567828be9be6
77096d9ccc9ec75d3208f20ac0dc8aa2
9b9cf3521280b06d13635d6b88ab741c
696cbda29ba3783e5673c6a31f964fe0
2b36c241e3a2b4466e208e3f4509e966
2a07f3c950608ce79f84a0bf5c8755e6
febad570ab62fe6b97e6d82bc469bb2e
0a88c34611792f5728ead7da54e2520f
92c312e48893f1b4073c36cba3de8dd0
ab7a1424b455c198f7caee65054a66b5
da2de8d54783fd9e0b5c1a0bf1d35ae5
941d63dd903114941e66b1ab373ef21c
8d76e8cd947b069ee7e10afb260b7c60
dbca9b3fe97f549b01589c3e14128d04
08f60019d8169f1d408d8407f3c8a7b5
eeaee4033e38a2c82ea8c1663fff6092
aa8a454a53d080fdc718190e80697d20
cba10d7100389117f5ad67b11ad971f3
709aea0ee4ad22cd54f0ab3a86875a91
d206b2cc53bdeb217ea4fe6ac5e44f92
cc6119b705e62f47e8bc76cc42ed6b72
fc844f51f5c3fa32dd7bbf738de0c36f
cbb16938a19c7f1298ba0c7101ea113b
50483660d99feb4968a6f5df144329a7
d115ccf6be8085183ccad415f89df059
3a6cddf2d754ed24e3aec10b747bf955
d17896eac4e11d0a806b4dd63f751da3
59cffa205f7bc30263d20c47e7b9a4df
d6fad96b3026474c3d1e4de2de10238a
60b07345c9cda19700ce6647688895a2
2dbbef2d1ab0340a19119f1761bbf229
a257bb0dda700d432124126c80b65553
31eef432ca3a7a8ccb22379c5165c698
dcd2a6043879ba5ef600808625448059
363aecc9ee7d93844c9d2580cf799765
cf89c3717b85542f0c27c2edcc7d4c1a
da982ab9a5fa38c77d73aef9db38bfbd
25e1d43d99c3a70b2273b49b3a977076
a6cf4e26357abefe0143a4b52984b137
239af31ae5919a77c04f75419a316da7
57db15fa393c0600206c9ae29051e40c
38750bfa17a90aade8cbde314f025e61
a967c10bab3c663347aac2483bf97e0e
562ba8c4c4208b32ee6794865cec5fff
e754fe0bb0947ff4dfef08d085791f9b
7e4dd77def253a583fab7149805f6f6a
0adef49e67b8d6a3a42139fc986d469e
2e2a4d7f98c29df37a1e32838a2992ea
b250014c1d91bf62fbaab36ae469c8ee
662286873f30cb1d8d0bcef3def3bad3
5deb611173115c81df17b1791d4290c7
d940164d61eecc7dea123a3f6559953e
e5ac8b8f12430e416f195369877fa1b2
7b79a200502ad10e2e4e4fc099ebdfd2
97c331e66f547973d93834d307690367
583faebe0f6952e886e6fab5a98b20fa
57b196bf5ac4d34668426161f7b17e75
2788cf7af4ff8148e8f2669ecca4cf2d
9ccfeb5be5a22eb81e15321876a7b71a
0375d28de6d4e686aa4f9ce870731ed6
7a95900e7940cca9ec6898a377a384b2
7ff590cdacff73ad504cacac48979707
3533c3fcd5d1c5300c32d8885da2a2f0
56afb3a73aee2d1544c43410146287aa
da64bf21ce94c975f2ceb3df5479692a
1337c27de8a23b3f9f715fd1348eef41
a3de2cd026ff0708d55ecf33f050325a
21a5952492fdcce62618b66c093cd4ae
dcfb3f8751ea97dc37a358fbf8d8481f
f72df9ee878bf330ba47a74490f1eb11
2cb08c233131277742e96b75f3c2a94a
3caaa52058d6977978e8b76ee8162040
079ecf92b2c72bd9148641dad5efe9b1
c9a539ad380e54eca50faf979f081e86
0356076ba28e8610998dad7ef0c77e0b
08592e148d20d6769c569437cbee0adb
ae6276d9f43a2b3353b6449b57f1fe27
ada279193e7116b9e773bf5aec3a9ad1
ef2b00e654ec25ec221c977cf168b0a0
bfcb0eb858a5799386d04316d5801b5e
af0953fc50173c463e0596ec33e9bb9b
792f27280f2888da9a8747989214d552
2a9cec5332add20b35bf7466df7461aa
e86d192d4dfbd227162160aafc9cbfc9
1c37213fddec3dd694391f9cae11b909
3c53ca14cb5c880338689a83e4a6a42d
0ec8b058d41f5029ad8a28b29a24f866
566245cfa707fa899275abe0c1f34b0e
882a617fbe8dd355978880f509ea3f17
0493a82acacaed9a226174c161956bca
12eadf1f65b52c87d938040b38e0013c
8c66383f6936bf9518df4b8d19ba9926
9513ce52ae8c647d02582e1e4fcc2cbb
9ec1f074f428094dbdb2c20fe51903c2
e70a10cec6052eaeca18eddf5bbf77e5
2cd6e23c1c41337604ad863645e1794a
bc3e607c9961e19fcc536576135ee260
062db34c9092894507b00b476ff6192b
6f356f3baea13392790de8aaba4e4573
bf44eceadd6908bf2bf0f8ce5bbf75b0
a26f3b50051a595af8162d4d67f52f90
0e8f299b93c84a52aa0823ce719963f9
abf9f8442362f0a7407f3e230544aff3
7a26d09ff2c08b14bec2c17a5d61e67d
6855ab0954de33d77064676936787488
b2984f26cd42106f26012fd17139f80d
19116eb28bc2bf6135bc684591269d9c
c032eb2baf87cce2772fd1bc6f62e7c0
c85f911282c79ade8da52f86820dd160
9df82598628fd0bdbcd073312495131c
7a51b123721d5deb1574e197fcc2ba5c
9b20c59a71ece5ec6d3b7f1a5c106c09
a14bab803e475d68c31dd13be2deb99e
b0118104e8a71e06576f302eeb80ea2e
ea153b34a308c14ec9b8ef228d9ed09f
7ba332b1b33c8efe1306944ae37c3556
00830094cdbad7386790acecc3059df9
2bff6f2a593fd027ecb9e83b12a00356
ac3a8fff17aee855423df4d962297736
864cc7d1acedad4313ccc6f8b5f89f93
ae4209b418e7cb3d7e565a97ca30e8a2
ae2106f1b2a83123afad222f8d6c99cd
0a720374961c0cfd7a236cb749aac572
30264c7dfe40d29075b6a24ef442f0ce
14fb07574ecc547bdc27940f6c34e0f0
9153d74d5f69a1244f5b4194ae9bc372
a3c710eb9fa00128bfb07d64287c9eb2
bdb4b2cec934f930e854a7f72a433210
91897b0eab55a95480288354e718b05d
9f986d582f608206f07b95f85d305be7
5fcd6cdb89c3928e3b045ca6ee6eefba
e126a0cdc0a639424939e41db3abd9de
772201440ce18175fcad96d4937dc834
efaa44e7207d50369ec068078a272826
7726eeca7ab1bfd1c8266b14b63318e8
dab1ed1ad45384ce9ab2fc2d5046dbe7
5b02040168d166fac7d30585955caeb8
055916c489888a6aefec2fe14f495104
beb7c522de21f3e525a95750019c063c
6151b4cb8fb4dc2321a6fe280270cd63
7900cf47fa7b40cc2e5518ebc4177b20
8c87caf17f0039dd69038d8f323fb809
f85175e990a06aeaafd2cbc4bdbf097a
8b7b1fe449d9dfc2315f2144eaac52e6
72cfd8d9654d80ae125ff00c3cf8e3e8
a67a66fca328465a99475de38ec137af
fd250e224aa34400a48f67c0eb070dba
48ed47e4166c0b158391cd34b5787d4e
91b979591b3ff783cc35a5f420d72abe
9fca04b47cd35bc5091a5bf818473cfd
9f4f0ebf8f07e411a00aa372f1871f24
83a9521d7979236906699d285c346b1e
242ab56369dd4c7042fff9ba47aaaddd
44a118a8fa6663f32c1eab2ee66073d5
c39f868d6f1111db746e21bcf9744787
a75bf0a9ec5feb237661599289d27f94
0d53cf64181d03e0d7bafb478820328f
ced13bc4584f6624cf7097a6ab4b0047
3be8ca6c0954c9b789a338884f358739
