=for gpg
-----BEGIN PGP SIGNED MESSAGE-----
Hash: SHA1

=head1 NAME

Iterator - A general-purpose iterator class, plus associated utility
functions.

=head1 VERSION

This documentation describes version 0.01 of Iterator.pm, July 26, 2005.

=cut

use strict;
use warnings;
package Iterator;
our $VERSION = '0.01';

use base 'Exporter';
use vars qw/@EXPORT @EXPORT_OK %EXPORT_TAGS/;

{   # encapsulation block to hide temporary variables.

@EXPORT  = qw();   # Nothing by default

my @util = qw(imap igrep irange ilist iarray ihead iappend
              ipairwise iskip iskip_until imesh izip iuniq);
my @file = qw(idir_listing idir_walk ifile ifile_reverse);
my @dbi  = qw(idb_rows);
my @misc = qw(ipermute igeometric inth irand_nth ifibonacci);
my @NOT  = qw(is_done);    # not part of any tag group

@EXPORT_OK   = (@EXPORT, @util, @file, @dbi, @misc, @NOT);
%EXPORT_TAGS = (
                all    => \@EXPORT_OK,
                util   => [@EXPORT, @util],
                file   => [@EXPORT, @file ],
                dbi    => [@EXPORT, @dbi  ],
                misc   => [@EXPORT, @misc ],
               );
}

# Declare exception classes
use Exception::Class
   (
    'Iterator::X' =>
    {
        description => 'Generic Iterator exception',
    },
    'Iterator::X::Parameter_Error' =>
    {
        isa         => 'Iterator::X',
        description => 'Iterator method parameter error',
    },
    'Iterator::X::Exhausted' =>
    {
        isa         => 'Iterator::X',
        description => 'Attempt to next_value () on an exhausted iterator',
    },
    'Iterator::X::Am_Now_Exhausted' =>
    {
        isa         => 'Iterator::X',
        description => 'Signals Iterator object that it is now exhausted',
    },
    'Iterator::X::User_Code_Error' =>
    {
        isa         => 'Iterator::X',
        fields      => 'eval_error',
        description => q{An exception was thrown within the user's code},
    },
    'Iterator::X::IO_Error' =>
    {
        isa         => 'Iterator::X',
        fields      => 'os_error',
        description => q{An I/O error occurred},
    },
    'Iterator::X::Internal_Error' =>
    {
        isa         => 'Iterator::X',
        description => 'An Iterator.pm internal error.  Please contact author.',
    },
   );

# Class method to help caller catch exceptions
sub Exception::Class::Base::caught
{
    my $class = shift;
    return Exception::Class->caught($class);
}

# Croak-like location of error
sub Iterator::X::location
{
    my ($pkg,$file,$line);
    my $caller_level = 0;
    while (1)
    {
        ($pkg,$file,$line) = caller($caller_level++);
        last if $pkg !~ /\A Iterator/x  &&  $pkg !~ /\A Exception::Class/x
    }
    return "at $file line $line";
}

# Die-like location of error
sub Iterator::X::Internal_Error::location
{
    my $self = shift;
    return "at " . $self->file () . " line " . $self->line ()
}

# Override full_message, to report location of error in caller's code.
sub Iterator::X::full_message
{
    my $self = shift;

    my $msg = $self->message;
    return $msg  if substr($msg,-1,1) eq "\n";

    $msg =~ s/[ \t]+\z//;   # remove any trailing spaces (is this necessary?)
    return $msg . q{ } . $self->location () . qq{\n};
}


## Constructor

# Method name:   new
# Synopsis:      $iterator = Iterator->new( $code_ref );
# Description:   Object constructor.
# Created:       07/27/2005 by EJR
# Parameters:    $code_ref - the iterator sequence generation code.
# Returns:       New Iterator.
# Exceptions:    Iterator::X::Parameter_Error (via _initialize)
sub new
{
    my $class = shift;
    my $self  = \do {my $anonymous};
    bless $self, $class;
    $self->_initialize(@_);
    return $self;
}

{ # encapsulation enclosure

    # Attributes:
    my %code_for;          # The sequence code (coderef) for each object.
    my %is_exhausted;      # Boolean: is this object exhausted?
    my %next_value_for;    # One-item lookahead buffer for each object.
    # [if you update this list of attributes, be sure to edit DESTROY]

    # Method name:   _initialize
    # Synopsis:      $iterator->_initialize( $code_ref );
    # Description:   Object initializer.
    # Created:       07/27/2005 by EJR
    # Parameters:    $code_ref - the iterator sequence generation code.
    # Returns:       Nothing.
    # Exceptions:    Iterator::X::Parameter_Error
    #                Iterator::X::User_Code_Error
    # Notes:         For internal module use only.
    #                Caches the first value of the iterator in %next_value_for.
    sub _initialize
    {
        my $self = shift;

        Iterator::X::Parameter_Error->throw(q{Too few parameters to Iterator->new()})
            if @_ < 1;
        Iterator::X::Parameter_Error->throw(q{Too many parameters to Iterator->new()})
            if @_ > 1;
        my $code = shift;
        Iterator::X::Parameter_Error->throw (q{Parameter to Iterator->new() must be code reference})
            if ref $code ne 'CODE';

        $code_for {$self} = $code;

        # Get the next (first) value for this iterator
        eval
        {
            $next_value_for{$self} = $code-> ();
        };

        my $ex;
        if ($ex = Iterator::X::Am_Now_Exhausted->caught ())
        {
            # Starting off exhausted is okay
            $is_exhausted{$self} = 1;
        }
        elsif ($@)
        {
            Iterator::X::User_Code_Error->throw (message => "$@",
                                                 eval_error => $@);
        }

        return;
    }

    # Method name:   DESTROY
    # Synopsis:      (none)
    # Description:   Object destructor.
    # Created:       07/27/2005 by EJR
    # Parameters:    None.
    # Returns:       Nothing.
    # Exceptions:    None.
    # Notes:         Invoked automatically by perl.
    #                Releases the hash entries used by the object.
    #                Module would leak memory otherwise.
    sub DESTROY
    {
        my $self = shift;
        delete $code_for{$self};
        delete $is_exhausted{$self};
        delete $next_value_for{$self};
    }

    # Method name:   value
    # Synopsis:      $next_value = $iterator->value();
    # Description:   Returns each value of the sequence in turn.
    # Created:       07/27/2005 by EJR
    # Parameters:    None.
    # Returns:       Next value, as generated by caller's code ref.
    # Exceptions:    Iterator::X::Exhausted
    # Notes:         Keeps one forward-looking value for the iterator in
    #                   %next_value_for.  This is so we have something to
    #                   return when user's code throws Am_Now_Exhausted.
    sub value
    {
        my $self = shift;

        Iterator::X::Exhausted->throw(q{Iterator is exhausted})
            if $is_exhausted{$self};

        # The value that we'll be returning this time.
        my $this_value = $next_value_for{$self};

        # Compute the value that we'll return next time
        eval
        {
            $next_value_for{$self} = $code_for{$self}->(@_);
        };
        if (my $ex = Iterator::X::Am_Now_Exhausted->caught ())
        {
            # Aha, we're done; we'll have to stop next time.
            $is_exhausted{$self} = 1;
        }
        elsif ($@)
        {
            Iterator::X::User_Code_Error->throw (message => "$@",
                                                 eval_error => $@);
        }

        return $this_value;
    }

    # Method name:   is_exhausted
    # Synopsis:      $boolean = $iterator->is_exhausted();
    # Description:   Flag indicating that the iterator is exhausted.
    # Created:       07/27/2005 by EJR
    # Parameters:    None.
    # Returns:       Current value of %is_exhausted for this object.
    # Exceptions:    None.
    sub is_exhausted
    {
        my $self = shift;

        return $is_exhausted{$self};
    }

    # Method name:   isnt_exhausted
    # Synopsis:      $boolean = $iterator->isnt_exhausted();
    # Description:   Flag indicating that the iterator is NOT exhausted.
    # Created:       07/27/2005 by EJR
    # Parameters:    None.
    # Returns:       Logical NOT of %is_exhausted for this object.
    # Exceptions:    None.
    sub isnt_exhausted
    {
        my $self = shift;

        return ! $is_exhausted{$self};
    }

} # end of encapsulation enclosure


# Function name: is_done
# Synopsis:      Iterator::is_done ();
# Description:   Convenience function. Throws an Am_Now_Exhausted exception.
# Created:       08/02/2005 by EJR, per Will Coleda's suggestion.
# Parameters:    None.
# Returns:       Doesn't return.
# Exceptions:    Iterator::X::Am_Now_Exhausted
sub is_done
{
    Iterator::X::Am_Now_Exhausted->throw()
}


# Function name: imap
# Synopsis:      $iter = imap {code} $another_iterator;
# Description:   Transforms an iterator.
# Created:       07/27/2005 by EJR
# Parameters:    code - Transformation code
#                $another_iterator - any other iterator.
# Returns:       Transformed iterator.
# Exceptions:    Iterator::X::Parameter_Error
#                Iterator::X::Am_Now_Exhausted
sub imap (&$)
{
    my ($transformation, $iter) = @_;

    Iterator::X::Parameter_Error->throw(q{Argument to imap must be an Iterator object})
        unless UNIVERSAL::isa($iter, 'Iterator');

    return Iterator->new( sub
    {
        Iterator::is_done if ($iter->is_exhausted);

        local $_ = $iter->value ();
        return $transformation-> ();
    });
}


# Function name: igrep
# Synopsis:      $iter = igrep {code} $another_iterator;
# Description:   Filters an iterator.
# Created:       07/27/2005 by EJR
# Parameters:    code - Filter condition.
#                $another_iterator - any other iterator.
# Returns:       Filtered iterator.
# Exceptions:    Iterator::X::Parameter_Error
#                Iterator::X::Am_Now_Exhausted
sub igrep (&$)
{
    my ($test, $iter) = @_;

    Iterator::X::Parameter_Error->throw(q{Argument to imap must be an Iterator object})
        unless UNIVERSAL::isa($iter, 'Iterator');

    return Iterator->new(sub
    {
        while ($iter->isnt_exhausted ())
        {
            local $_ = $iter->value ();
            return $_ if $test-> ();
        }

        Iterator::is_done();
    });
}


# Function name: irange
# Synopsis:      $iter = irange ($start, $end, $step);
# Description:   Generates an arithmetic sequence of numbers.
# Created:       07/27/2005 by EJR
# Parameters:    $start - First value.
#                $end   - Final value.     (may be omitted)
#                $step  - Increment value. (may be omitted)
# Returns:       Sequence iterator
# Exceptions:    Iterator::X::Am_Now_Exhausted
# Notes:         If the $end value is omitted, iterator is unbounded.
#                If $step is omitted, it defaults to 1.
#                $step may be negative (or even zero).
sub irange
{
    my ($from, $to, $step) = @_;
    $step = 1 unless defined $step;

    return Iterator->new (sub
    {
        # Reached limit?
        Iterator::is_done
                if (defined($to)
                    &&  ($step>0 && $from>$to  ||  $step<0 && $from<$to) );

        # This iteration's return value
        my $retval = $from;

        $from += $step;
        return $retval;
    });
}

# Function name: ilist
# Synopsis:      $iter = ilist (@list);
# Description:   Creates an iterator from a list
# Created:       07/28/2005 by EJR
# Parameters:    @list - list of values to iterate over
# Returns:       Array (list) iterator
# Exceptions:    Iterator::X::Am_Now_Exhausted
# Notes:         Makes an internal copy of the list.
sub ilist
{
    my @items = @_;
    my $index=0;
    return Iterator->new( sub
    {
        Iterator::is_done if ($index >= @items);
        return $items[$index++];
    });
}

# Function name: iarray
# Synopsis:      $iter = iarray ($a_ref);
# Description:   Creates an iterator from an array reference
# Created:       07/28/2005 by EJR
# Parameters:    $a_ref - Reference to array to iterate over
# Returns:       Array iterator
# Exceptions:    Iterator::X::Parameter_Error
#                Iterator::X::Am_Now_Exhausted
# Notes:         Does not make an internal copy of the list.
sub iarray ($)
{
    my $items = shift;
    my $index=0;

    Iterator::X::Parameter_Error->throw->
        (q{Argument to iarray must be an array reference})
            if ref $items ne 'ARRAY';

    return Iterator->new( sub
    {
        Iterator::is_done if $index >= @$items;
        return $items->[$index++];
    });
}

# Function name: ihead
# Synopsis:      $iter = ihead ($num, $some_other_iterator);
# Synopsis:      @valuse = ihead ($num, $iterator);
# Description:   Returns at most $num items from other iterator.
# Created:       07/28/2005 by EJR
#                08/02/2005 EJR: combined with ahead, per Will Coleda
# Parameters:    $num - Max number of items to return
#                $some_other_iterator - another iterator
# Returns:       limited iterator
# Exceptions:    Iterator::X::Parameter_Error
#                Iterator::X::Am_Now_Exhausted
sub ihead
{
    my $num  = shift;
    my $iter = shift;

    Iterator::X::Parameter_Error->throw
        (q{Second parameter for ihead must be an Iterator})
            unless UNIVERSAL::isa($iter, 'Iterator');

    # List context?  Return the first $num elements.
    if (wantarray)
    {
        my @a;
        while ($iter->isnt_exhausted  &&  (!defined($num)  ||  $num-- > 0))
        {
            push @a, $iter->value;
        }
        return @a;
    }

    # Scalar context: return an iterator to return at most $num elements.
    return Iterator->new(sub
    {
        Iterator::is_done if $num <= 0;

        $num--;
        return $iter->value;
    });
}

# Function name: iappend
# Synopsis:      $iter = iappend (@iterators);
# Description:   Joins a bunch of iterators together.
# Created:       07/28/2005 by EJR
# Parameters:    @iterators - any number of other iterators
# Returns:       A "merged" iterator.
# Exceptions:    Iterator::X::Parameter_Error
#                Iterator::X::Am_Now_Exhausted
sub iappend
{
    my @its = @_;

    # Check types
    foreach (@its)
    {
        Iterator::X::Parameter_Error->throw
            (q{All parameters for iarray must be Iterators})
                unless UNIVERSAL::isa($_, 'Iterator');
    }

    # Passthru, if there's only one.
    return $its[0] if @its == 1;

    return Iterator->new (sub
    {
        my $val;

        # Any empty iterators at front of list?  Remove'em.
        while (@its  &&  $its[0]->is_exhausted)
        {
            shift @its;
        }

        # No more iterators?  Then we're done.
        Iterator::is_done
            if @its == 0;

        # Return the next value of the iterator at the head of the list.
        return $its[0]->value;
    });
}

# Function name: ipairwise
# Synopsis:      $iter = ipairwise {code} ($iter1, $iter2);
# Description:   Applies an operation to pairs of values from iterators.
# Created:       07/28/2005 by EJR
# Parameters:    code - transformation, may use $a and $b
#                $iter1 - First iterator; "$a" value.
#                $iter2 - First iterator; "$b" value.
# Returns:       Iterator
# Exceptions:    Iterator::X::Parameter_Error
#                Iterator::X::Am_Now_Exhausted
sub ipairwise (&$$)
{
    my $op    = shift;
    my $iterA = shift;
    my $iterB = shift;

    # Check types
    for ($iterA, $iterB)
    {
        Iterator::X::Parameter_Error->throw
            (q{Second and third parameters for ipairwise must be Iterators})
                unless UNIVERSAL::isa($_, 'Iterator');
    }

    return Iterator->new(sub
    {
        Iterator::is_done
            if $iterA->is_exhausted  ||  $iterB->is_exhausted;

        # Localize $a and $b
        # My thanks to Benjamin Goldberg for this little bit of evil.
        my ($caller_a, $caller_b) = do
        {
            my $pkg;
            my $i = 1;
            while (1)
            {
                $pkg = caller($i++);
                last if $pkg ne 'Iterator';
            }
            no strict 'refs';
            \*{$pkg.'::a'}, \*{$pkg.'::b'};
        };

        # Set caller's $a and $b
        local (*$caller_a, *$caller_b) = \($iterA->value, $iterB->value);

        # Invoke caller's operation
        return $op->();
    });
}

# Function name: iskip
# Synopsis:      $iter = iskip $num, $another_iterator
# Description:   Skips the first $num values of another iterator
# Created:       07/28/2005 by EJR
# Parameters:    $num - how many values to skip
#                $another_iterator - another iterator
# Returns:       Sequence iterator
# Exceptions:    None
sub iskip
{
    my $num = shift;
    my $it  = shift;

    Iterator::X::Parameter_Error->throw
        (q{Second parameter for iskip must be an Iterator})
            unless UNIVERSAL::isa($it, 'Iterator');

    # Discard first $num values
    $it->value  while $it->isnt_exhausted  &&  $num-->0;

    return $it;
}


# Function name: iskip_until
# Synopsis:      $iter = iskip_until {code}, $another_iterator
# Description:   Skips values of another iterator until {code} is true.
# Created:       07/28/2005 by EJR
# Parameters:    {code} - Determines when to start returning values
#                $another_iterator - another iterator
# Returns:       Sequence iterator
# Exceptions:    Iterator::X::Am_Now_Exhausted
sub iskip_until (&$)
{
    my $code = shift;
    my $iter = shift;
    my $value;
    my $found_it = 0;

    Iterator::X::Parameter_Error->throw
        (q{Second parameter for iskip_until must be an Iterator})
            unless UNIVERSAL::isa($iter, 'Iterator');

    # Discard first $num values
    while ($iter->isnt_exhausted)
    {
        local $_ = $iter->value;
        if ($code->())
        {
            $found_it = 1;
            $value = $_;
            last;
        }
    }

    # Didn't find it?  Pity.
    Iterator::is_done
        unless $found_it;

    # Return an iterator with this value, and all remaining values.
    return iappend ilist($value), $iter;
}


# Function name: idir_listing
# Synopsis:      $iter = idir_listing ($path)
# Description:   Returns the full file names in the specified directory.
# Created:       07/28/2005 by EJR
# Parameters:    $path - Directory.  If omitted, uses current dir.
# Returns:       Iterator
# Exceptions:    Iterator::X::Am_Now_Exhausted
sub idir_listing
{
    require IO::Dir;
    require Cwd;

    my $path = shift || Cwd::getcwd();
    $path =~ s|/ \z||x;   # remove any trailing slash
    my $d = new IO::Dir $path;
    Iterator::X::IO_Error (message => qq{Cannot read "$path": $!},
                           error => $!)
        unless $d;

    return Iterator->new (sub
    {
        # Get next file, skipping . and ..
        my $next;
        while (1)
        {
            $next = $d->read;

            if (! defined $next)
            {
                undef $d;   # allow garbage collection
                Iterator::is_done;
            }

            last  if $next ne '.'  &&  $next ne '..';
        }

        # Return the filename
        return "$path/$next";
    });
}


# Function name: idir_walk
# Synopsis:      $iter = idir_walk ($path)
# Description:   Returns the directory tree below a given dir.
# Created:       07/28/2005 by EJR
# Parameters:    $path - Directory.  If omitted, uses current dir.
# Returns:       Iterator
# Exceptions:    Iterator::X::Am_Now_Exhausted
sub idir_walk
{
    my @dir_queue;
    my $path = shift;
    my $files = idir_listing($path);

    return Iterator->new (sub
    {
        # If no more files in current directory,
        # get next directory off the queue
        while ($files->is_exhausted)
        {
            # Nothing else on the queue?  Then we're done.
            if (@dir_queue == 0)
            {
                undef $files;    # allow garbage collection
                Iterator::is_done;
            }

            # Create an iterator to return the files in that directory
            $files = idir_listing(shift @dir_queue);
        }

        # Get next file in current directory
        my $next = $files->value;

        # If this is a directory (and not a symlink), remember it for later recursion
        if (-d $next  &&  !-l $next)
        {
            unshift @dir_queue, $next;
        }

        return $next;
    });
}

# Function name: ifile
# Synopsis:      $iter = ifile ($filename)
# Description:   Returns the lines of a file, one at a time.
# Created:       07/28/2005 by EJR
# Parameters:    $filename - File name to open.
# Returns:       Iterator
# Exceptions:    Iterator::X::Parameter_Error
#                Iterator::X::IO_Error
#                Iterator::X::Am_Now_Exhausted
sub ifile
{
    require IO::File;

    my $filename  = shift;
    my $autochomp = shift;
    my $sep       = @_ ? shift : $/;

    if (defined $autochomp)
    {
        Iterator::X::Parameter_Error->throw(q{Invalid "chomp" argument to ifile})
            if ($autochomp ne 'chomp'  &&  $autochomp ne 'nochomp');
    }
    else
    {
        $autochomp = 'chomp';    # default
    }

    # Open the file handle.
    my $fh = new IO::File ($filename);
    Iterator::X::IO_Error (message => qq{Cannot read "$filename": $!},
                           error => $!)
        unless $fh;

    return Iterator->new (sub
    {
        my $line;

        # Get next line (delimited by $sep);
        {
            local $/ = $sep;
            $line = $fh->getline();
            chomp $line  if defined $line  &&  $autochomp eq 'chomp';
        }

        # Done?
        if (!defined $line)
        {
            $fh->close;
            undef $fh;
            Iterator::is_done;
        }

        # Return the line
        return $line;
    });
}

# Function name: ifile_reverse
# Synopsis:      $iter = ifile_reverse ($filename)
# Description:   Returns the lines of a file, in reverse order
# Created:       07/28/2005 by EJR
# Parameters:    $filename - File name to open.
# Returns:       Iterator
# Exceptions:    Iterator::X::Parameter_Error
#                Iterator::X::IO_Error
#                Iterator::X::Am_Now_Exhausted
sub ifile_reverse
{
    require IO::File;

    my $filename  = shift;
    my $autochomp = shift;
    my $sep  = @_? shift : $/;

    if (defined $autochomp)
    {
        Iterator::X::Parameter_Error->throw(q{Invalid "chomp" argument to ifile_reverse})
            if ($autochomp ne 'chomp'  &&  $autochomp ne 'nochomp');
    }
    else
    {
        $autochomp = 'chomp';    # default
    }

    # Must read chunks of the end of the file into memory
    my $block_size   = shift || 8192;

    my $fh = new IO::File $filename;
    Iterator::X::IO_Error (message => qq{Cannot read "$filename": $!},
                           error => $!)
        unless $fh;

    # Buffer variables
    my $leftover;
    my @lines;

    # Are we at the start of the file?
    my $at_start = sub {$fh->tell == 0};

    my $break = sub
    {
        my $block = shift;
        $block .= $leftover if defined $leftover;
        @lines = reverse split /(?<=\Q$sep\E)/, $block;
        $leftover = pop @lines;
    };

    my $prev_block = sub
    {
        my $pos = $fh->tell;
        my $bytes = 1 + ($pos-1) % $block_size;
        my $buf;

        my $seek_ok = seek $fh, -$bytes, 1;
        Iterator::X::IO_Error->throw
                (message => qq{Seek error on $filename: $!},
                 os_error => $!)
            unless $seek_ok;

        my $num_read = read $fh, $buf, $bytes;
        Iterator::X::IO_Error->throw
                (message => qq{Read error on $filename: $!},
                 os_error => $!)
            if ! defined $num_read;

        seek $fh, -$bytes, 1;
        Iterator::X::IO_Error->throw
                (message => qq{Seek error on $filename: $!},
                 os_error => $!)
            unless $seek_ok;

        return $buf;
    };

    seek $fh, 0, 2;    # end of file
    $break->( $prev_block->() );

    return Iterator->new (sub
    {
        if (@lines == 0)
        {
            if ($at_start->())
            {
                @lines = $leftover;
                undef $leftover;
            }
            else
            {
                $break->( $prev_block->() );
            }
        }

        # Return the line (chomped if so requested)
        my $line = shift @lines;

        # Exhausted?
        Iterator::is_done
            if ! defined $line;

        $line =~ s/\Q$sep\E$//  if $autochomp eq 'chomp';
        return $line;
    });
}

# Function name: idb_rows
# Synopsis:      $iter = idb_rows ($dbh, $sql, @bind_vars);
# Description:   Iterates over a database query's results
# Created:       07/29/2005 by EJR
# Parameters:    $dbh - A DBI database handle
#                $sql - The query
#                @bind_vars - (optional) bind variables.
# Returns:       Row iterator (returns hash references)
# Exceptions:    Iterator::X::Parameter_Error
#                "idb_rows cannot prepare sql: <error string>"
#                "idb_rows cannot execute sql: <error string>"
#                "fetchrow_hashref: <error string>"
#                Iterator::X::Am_Now_Exhausted
sub idb_rows
{
    my ($dbh, $sql, @bind) = @_;
    my $sth;    # statement handle

    Iterator::X::Parameter_Error->throw
        ('idb_rows: $dbh parameter is not a database handle')
        unless UNIVERSAL::can($dbh, 'prepare');

    return Iterator->new (sub
    {
        # Prepare database statement, if not done alread
        unless ($sth)
        {
            $sth = $dbh->prepare($sql)
                or die "idb_rows cannot prepare sql: " . $dbh->errstr;

            unless ($sth->execute(@bind))
            {
                $sth->finish;
                undef $sth;     # allow garbage collection
                die "idb_rows cannot execute sql: " . $sth->errstr;
            }
        }

        # Fetch the row
        my $row_ref = $sth->fetchrow_hashref;

        # Check for errors
        if (!defined $row_ref)
        {
            if ($sth->err)
            {
                die "fetchrow_hashref: " . $sth->errstr;
            }
            Iterator::is_done;
        }

        return $row_ref;
    });
}

# Function name: ipermute
# Synopsis:      $iter = ipermute (@items);
# Description:   Generates permutations of a list.
# Created:       07/29/2005 by EJR
# Parameters:    @items - the items to be permuted.
# Returns:       Sequence iterator
# Exceptions:    Iterator::X::Am_Now_Exhausted
# Notes:         Algorithm from MJD's book.
sub ipermute
{
    my @items = @_;
    my @digits = (0) x @items;     # "Odometer".  See Dominus, pp 128-135.

    return Iterator->new (sub
    {
        unless (@digits)
        {
            Iterator::is_done;
        }

        # Use the existing state to create a new permutation
        my @perm = ();
        my @c_items = @items;
        push @perm, splice(@c_items, $_, 1)  for @digits;

        # Find the rightmost column that isn't already maximum
        my $column = $#digits;
        until ($digits[$column] < $#digits-$column || $column < 0)
            { $column-- }

        if ($column < 0)
        {
            # Last set. Generate no more.
            @digits = ();
        }
        else
        {
            # Increment the rightmost column; set colums to the right to zeroes
            $digits[$column]++;
            $digits[$_] = 0  for ($column+1 .. $#digits);
        }

        return \@perm;
    });
}


# Function name: ifibonacci
# Synopsis:      $iter = ifibonacci ($start1, $start2);
# Description:   Generates a Fibonacci sequence.
# Created:       07/27/2005 by EJR
# Parameters:    $start1 - First starting value
#                $start2 - Second starting value
# Returns:       Sequence iterator
# Exceptions:    Iterator::X::Am_Now_Exhausted
# Notes:         If $start2 is omitted, $start1 is used for both.
#                If both are omitted, 1 is used for both.
sub ifibonacci
{
    my ($start1, $start2) = @_ == 0?  (1, 1)
                          : @_ == 1?  ($_[0], $_[0])
                          : @_ == 2?  @_
                          : Iterator::X::Parameter_Error->throw
                              ("Too many arguments to ifibonacci");

    return Iterator->new( sub
    {
        my $retval;
        ($retval, $start1, $start2) = ($start1, $start2, $start1+$start2);
        return $retval;
    });
}

# Function name: igeometric
# Synopsis:      $iter = igeometric ($start, $end, $factor);
# Description:   Generates a geometric sequence.
# Created:       07/28/2005 by EJR
# Parameters:    $start - Starting value
#                $end - Ending value
#                $factor - multiplier.
# Returns:       Sequence iterator
# Exceptions:    Iterator::X::Am_Now_Exhausted
# Notes:         If $end if omitted, series is unbounded.
#                $factor must be specified.
sub igeometric
{
    my ($start, $end, $factor) = @_;
    my $growing = abs($factor) >= 1;

    return Iterator->new (sub
    {
        Iterator::is_done
            if (defined $end  &&  ($growing && $start > $end  ||  !$growing && $start < $end));

        my $retval = $start;
        $start *= $factor;
        return $retval;
    });
}

# Function name: inth
# Synopsis:      $iter = inth ($n, $iter)
# Description:   Returns 1 out of every $n items.
# Created:       07/29/2005 by EJR
# Parameters:    $n - frequency
#                $iter - other iterator
# Returns:       Sequence iterator
# Exceptions:    Iterator::X::Parameter_Error
#                Iterator::X::Am_Now_Exhausted
sub inth
{
    my $n1 = -1 + shift;
    my $iter = shift;

    Iterator::X::Parameter_Error->throw('Invalid "$n" value for inth')
        if $n1 < 0;

    Iterator::X::Parameter_Error->throw
        (q{Second parameter for inth must be an Iterator})
            unless UNIVERSAL::isa($iter, 'Iterator');

    return Iterator->new (sub
    {
        my $i = $n1;
        while ($i-->0  &&  $iter->isnt_exhausted)
        {
            $iter->value();   # discard value
        }

        Iterator::is_done
            if $iter->is_exhausted;

        return $iter->value();
    });
}

# Function name: irand_nth
# Synopsis:      $iter = irand_nth ($n, $iter)
# Description:   Returns 1 out of every $n items, randomly.
# Created:       07/29/2005 by EJR
# Parameters:    $n - frequency
#                $iter - other iterator
# Returns:       Sequence iterator
# Exceptions:    Iterator::X::Parameter_Error
#                Iterator::X::Am_Now_Exhausted
sub irand_nth
{
    my $n    = shift;
    my $iter = shift;

    Iterator::X::Parameter_Error->throw('Invalid "$n" value for inth')
        if $n <= 0;

    Iterator::X::Parameter_Error->throw
        (q{Second parameter for irand_nth must be an Iterator})
            unless UNIVERSAL::isa($iter, 'Iterator');

    my $prob = 1 / $n;

    return Iterator->new (sub
    {
        while (rand > $prob  &&  $iter->isnt_exhausted)
        {
            $iter->value();   # discard value
        }

        Iterator::is_done
            if $iter->is_exhausted;

        return $iter->value();
    });
}

# Function name: imesh / izip
# Synopsis:      $iter = imesh ($iter1, $iter2, ...)
# Description:   Merges other iterators together.
# Created:       07/30/2005 by EJR
# Parameters:    Any number of other iterators.
# Returns:       Sequence iterator
# Exceptions:    Iterator::X::Parameter_Error
#                Iterator::X::Am_Now_Exhausted
foreach my $sub (qw/imesh izip/)
{
    no strict 'refs';
    *$sub = sub
    {
        use strict 'refs';

        my @iterators = @_;
        my $it_index  = 0;

        foreach my $iter (@iterators)
        {
            Iterator::X::Parameter_Error->throw(
                "Argument to $sub is not an iterator")
                unless UNIVERSAL::isa($iter, 'Iterator');
        }

        return Iterator->new (sub
        {
            Iterator::is_done
                if is_exhausted($iterators[$it_index]);

            my $retval = $iterators[$it_index]->value();

            if (++$it_index >= @iterators)
            {
                $it_index = 0;
            }

            return $retval;
        });
    };
}

# Function name: iuniq
# Synopsis:      $iter = iuniq ($another_iterator);
# Description:   Removes duplicate entries from an iterator.
# Created:       07/30/2005 by EJR
# Parameters:    Another iterator.
# Returns:       Sequence iterator
# Exceptions:    Iterator::X::Parameter_Error
#                Iterator::X::Am_Now_Exhausted
sub iuniq
{
    Iterator::X::Parameter_Error->throw ("Too few parameters to iuniq")
        if @_ < 1;
    Iterator::X::Parameter_Error->throw ("Too many parameters to iuniq")
        if @_ > 1;

    my $iter = shift;
    Iterator::X::Parameter_Error->throw("Argument to iuniq is not an iterator")
        unless UNIVERSAL::isa($iter, 'Iterator');

    my %did_see;
    return Iterator->new (sub
    {
        my $value;
        while (1)
        {
            Iterator::is_done
                if $iter->is_exhausted;

            $value = $iter->value;
            last if !$did_see{$value}++;
        }
        return $value;
    });
}


1;
__END__

=head1 SYNOPSIS I: class basics

 use Iterator;

 # Making your own iterators from scratch:
 $iterator = Iterator->new ( sub { code } );

 # Accessing an iterator's values in turn:
 $next_value = $iterator->value();

 # Is the iterator out of values?
 $boolean = $iterator->is_exhausted();
 $boolean = $iterator->isnt_exhausted();

 # Within {code}, above:
 Iterator::is_done();    # to signal end of sequence.

=head1 SYNOPSIS II: essential utility functions

 use Iterator qw(:util);

 # Transform sequences
 $iterator = imap { transformation code } $some_other_iterator;

 # Filter sequences
 $iterator = igrep { condition code } $some_other_iterator;

 # Range of values  (arithmetic sequence)
 $iter = irange ($start, $end, $increment);
 $iter = irange ($start, $end);
 $iter = irange ($start);

 # Iterate over an arbitrary list
 $iter = ilist (item, item, ...);
 $iter = ilist (@items);

 # Iterate over an array, by reference
 $iter = iarray (\@array);

 # Return at most $num items from an iterator
 $iter   = ihead ($num, $some_other_iterator);
 @values = ihead ($num, $some_other_iterator);

 # Append multiple iterators into one
 $iter = iappend ($it1, $it2, $it3, ...);

 # Apply a function to pairs of iterator values
 $iter = ipairwise {code} $iter_A, $iter_B;

 # Skip the first $num values of an iterator
 $iter = iskip ($num, $some_other_iterator);

 # Skip values from an iterator until a condition is met
 $iter = iskip_until {code} $some_other_iterator;

=head1 SYNOPSIS III: filesystem iterator functions

 use Iterator qw(:file);

 # Return the names of files in a directory (except . and ..)
 $iter = idir_listing ($path);

 # Return all the files in a directory tree, one at a time.
 # Like File::Find, in slow motion.
 $iter = idir_walk ($path);

 # Return the lines of a file, one at a time.
 $iter = ifile ($filename, [$chomp], [$sep]);

 # Return the lines of a file, in reverse order
 $iter = ifile_reverse ($filename, [$chomp], [$sep]);

=head1 SYNOPSIS III: Database iteration

 use Iterator qw(:dbi);

 # Iterate over a database SELECT query.
 # (returns one hash reference per row).
 # (requires DBI).
 $iter = idb_rows ($dbh, $sql);
 $iter = idb_rows ($dbh, $sql, @bind);

=head1 SYNOPSIS IV: Miscellaneous iterators

 use Iterator qw(:misc);

 # Permute the elements of a list:
 $iter = ipermute (@items);

 # Select only every nth value of an iterator
 $iter = inth ($n, $another_iterator);

 # Randomly select iterator values with 1/$n probability
 $iter = irand_nth ($n, $another_iterator);

 # Fibonacci sequence
 $ifib = ifibonacci();         # default sequence starts with 1,1
 $ifib = ifibonacci($a, $b);   # or specify alternate starting pair

 # Geometric sequence
 $iter = igeometric ($start, $end, $multiplier);

 # Mesh iterators together
 $iter = imesh ($iter, $iter, ...);
 $iter = izip  ($iter, $iter, ...);

 # Return each value of an iterator once
 $iter = iuniq ($another_iterator);

=head1 DESCRIPTION

This module is meant to be the definitive implementation of iterators,
as popularized by Mark Jason Dominus's lectures and recent book
(_Higher Order Perl_, Morgan Kauffman, 2005).

An "iterator" is an object, represented as a code block that generates
the "next value" of a sequence, and generally implemented as a
closure.  When you need a value to operate on, you pull it from the
iterator.  If it depends on other iterators, it pulls values from them
when it needs to.  Contrast this with ordinary array processing, where
you load or compute all of the input values at once, then loop over
them in memory.  It's analogous to the difference between looping over
a file one line at a time, and reading the entire file into an array
of lines before operating on it.

Iterator.pm provides a class that simplifies creation and use of these
iterator objects, and provides many general-purpose iterator
functions.

Some iterators are infinite (that is, they generate infinite
sequences), and some are finite.  When the end of a finite sequence is
reached, the iterator code block should throw an exception of the type
C<Iterator::X::Am_Now_Exhausted>.  This will signal the Iterator class
to mark the object as exhausted.  The C<is_exhausted> method will then
return true, and the C<isnt_exhausted> method will return false.  Any
further calls to the C<value> method will throw an exception of the
type C<Iterator::X::Exhausted>.  See L<DIAGNOSTICS>.

Note that in many, many cases, you will not need to explicitly create
an iterator; there are plenty of iterator generation and manipulation
functions included in this module.  You can just plug them together
like building blocks.

=head1 METHODS

=over 4

=item new

 $iter = Iterator->new( sub { code } );

Creates a new iterator object.  The code block that you provide will
be invoked by the C<value> method.  The code block should have some
way of maintaining state, so that it knows how to return the next
value of the sequence each time it is called.

If the code is called after it has generated the last value in its
sequence, it should throw an exception:

    Iterator::X::Am_Now_Exhausted->throw ();

This is such a common need that there is a convenience function for
it:

    Iterator::is_done ();

=item value

 $next_value = $iter->value ();

Returns the next value in the iterator's sequence.  If C<value> is
called on an exhausted iterator, an Iterator::X::Exhausted exception
is thrown.

=item is_exhausted

 $bool = $iter->is_exhausted ();

Returns true if the iterator is exhausted.  In this state, any call
to the iterator's C<value> method will throw an exception.

=item isnt_exhausted

 $bool = $iter->isnt_exhausted ();

Returns true if the iterator is not yet exhausted.

=back

=head1 FUNCTIONS

=over 4

=item imap

 $iter = imap { transformation code } $some_other_iterator;

Returns an iterator that is a transformation of some other iterator.
Within the transformation code, C<$_> is set to each value of the
other iterator, in turn.

I<Examples:>

 $evens   = imap { $_ * 2  }  irange (0);  # returns 0, 2, 4, ...
 $squares = imap { $_ * $_ }  irange (7);  # 49, 64, 81, 100, ...

=item igrep

 $iter = igrep { condition } $some_other_iterator;

Returns an iterator that selectively returns values from some other
iterator.  Within the C<condition> code, C<$_> is set to each value of
the other iterator, in turn.

I<Examples:>

 $fives = igrep { $_ % 5 == 0 } irange (0,10);   # returns 0, 5, 10
 $small = igrep { $_ < 10 }     irange (8,12);   # returns 8, 9

=item irange

 $iter = irange ($start, $end, $increment);
 $iter = irange ($start, $end);
 $iter = irange ($start);

C<irange> returns a sequence of numbers.  The sequence begins with
C<$start>, ends at C<$end>, and steps by C<$increment>.  This is sort
of the Iterator version of a C<for> loop.

If C<$increment> is not specified, 1 is used.  C<$increment> may be
negative -- or even zero, in which case iterator returns an infinite
sequence of C<$start>.

If C<$end> is not specified (is C<undef>), the sequence is infinite.

I<Examples:>

 $iter = irange (1, 2);           #  Iterate from 1 to 2
 $val  = $iter->value();          #  $val is now 1.
 $val  = $iter->value();          #  $val is now 2.
 $bool = $iter->is_exhausted();   #  $bool is now true.

 $iter = irange (10, 8, -1);      #  Iterate from 10 down to 8
 $iter = irange (1);              #  Iterate from 1, endlessly.

=item ilist

 $iter = ilist (@items);

Returns an iterator that iterates over an arbitrary sequence of
values.  It's sort of an Iterator version of C<foreach>.

This function makes an internal copy of the list, so it may not be
appropriate for an extremely large list.

I<Example:>

 $iter = ilist (4, 'minus five', @foo, 7);
 $val  = $iter->value();          # $val is now 4
 $val  = $iter->value();          # $val is now 'minus five'
 ...

=item iarray

 $iter = iarray (\@array);

Returns an iterator that iterates over an array.  Note that since it
uses a reference to that array, if you modify the array, that will be
reflected in the values returned by the iterator.  This may be What
You Want.  Or it may cause Hard-To-Find Bugs.

=item ihead

 $iter   = ihead ($num, $some_other_iterator);
 @values = ihead ($num, $some_iterator);

In scalar context, creates an iterator that returns at most C<$num>
items from another iterator, then stops.

In list context, returns the first C<$num> items from the iterator.
If C<$num> is C<undef>, all remaining values are pulled
from the iterator until it is exhausted.  Use C<undef> with caution;
iterators can be huge -- or infinite.

I<Examples:>

 $iota5 = ihead 5, irange 1;    # returns 1, 2, 3, 4, 5.

 $iter = irange 1;            # infinite sequence, starting with 1
 @vals = ihead (5, $iter);    # @vals is (1, 2, 3, 4, 5)
 $nextval = $iter->value;     # $nextval is 6.

=item iappend

 $iter = iappend (@list_of_iterators);

Creates an iterator that consists of any number of other iterators
glued together.  The resulting iterator pulls values from the first
iterator until it's exhausted, then from the second, and so on.

=item ipairwise

 $iter = ipairwise {code} $it_A, $it_B;

Creates a new iterator which applies C<{code}> to pairs of elements of
two other iterators, C<$it_A> and C<$it_B> in turn.  The pairs are
assigned to C<$a> and C<$b> before invoking the code.

The new iterator is exhausted when either C<$it_A> or C<$it_B> are
exhausted.

This function is analogous to the C<pairwise> function from List::MoreUtils.

I<Example:>

 $first  = irange 1;                              # 1,  2,  3,  4, ...
 $second = irange 4, undef, 2;                    # 4,  6,  8, 10, ...
 $third  = ipairwise {$a * $b} $first, $second;   # 4, 12, 24, 40, ...

=item iskip

 $iter = iskip ($num, $another_iterator);

Returns an iterator that contains the values of C<$another_iterator>,
minus the first C<$num> values.  In other words, skips the first
C<$num> values of C<$another_iterator>.

I<Example:>

 $iter = ilist (24, -1, 7, 8);        # Bunch of random values
 $cdr  = iskip 1, $iter;              # "pop" the first value
 $val  = $cdr->value();               # $val is -1.

=item iskip_until

 $iter = iskip_until {code} $another_iterator;

Returns an iterator that skips the leading values of C<$another_iterator>
until C<{code}> evaluates to true for one of its values.  C<{code}>
can refer to the current value as C<$_>.

I<Example:>

 $iter = iskip_until {$_ > 5}  irange 1;    # returns 6, 7, 8, 9, ...

=item idir_listing

 $iter = idir_listing ($path);

Iterator that returns the names of the files in the C<$path>
directory.  If C<$path> is omitted, defaults to the current directory.
Does not return the C<.> and C<..> files (under unix).

Requires IO::Dir and Cwd.

I<Example:>

To return only certain files, combine this with an igrep:

 $iter = igrep {-s && -M < 1} idir "/some/path";

(Returns non-empty files modified less than a day ago).

=item idir_walk

 $iter = idir_walk ($path);

Returns the files in a directory tree, one by one.  Like File::Find in
slow motion.

Requires IO::Dir and Cwd.

=item ifile

 $iter = ifile ($filename, [$chomp], [$sep]);

Opens a file, generates an iterator to return the lines of the file.

If C<$chomp> is passed, it should be 'C<chomp>' or 'C<nochomp>', to
indicate whether lines should be chomped on input.  The default (if
C<$chomp> is undefined) is 'C<chomp>'.

C<$sep>, if passed, is the record separator.

Requires IO::File.

=item ifile_reverse

 $iter = ifile_reverse ($filename, [$chomp], [$sep]);

Exactly the same as ifile, but reads the lines of the file backwards.

=item idb_rows

 $it = idb_rows ($dbh, $sql);
 $it = idb_rows ($dbh, $sql, @bind);

Returns an iterator to return rows from a database query.  Each row is
returned as a hashref, as from C<fetchrow_hashref> from the DBI
module.

If the query requires bind variables, they may be passed in C<@bind>.

I<Example:>

 $dbh = DBI->connect (...);
 $iter = idb_rows ($dbh, 'select foo, bar from quux');
 $row_ref = $iter->value;

=item ipermute

 $iter = ipermute (@list);
 $array_ref = $iter->value();

Permutes the items in an arbitrary list.  Each time the iterator is
called, it returns the next combination of the items, in the form of a
reference to an array.

I<Example:>

 $iter = ipermute ('one', 'two', 'three');
 $ref  = $iter->value();          # -> ['one', 'two', 'three']
 $ref  = $iter->value();          # -> ['one', 'three', 'two']
 $ref  = $iter->value();          # -> ['two', 'one', 'three']
 # ...etc

=item inth

 $iter = inth ($n, $another_iterator);

Returns an iterator to return every I<nth> value from the input
iterator.  The first C<$n-1> items are skipped, then one is returned,
then the next C<$n-1> items are skipped, and so on.

This can be useful for sampling data.

=item irand_nth

 $iter = irand_nth ($n, $another_iterator);

Random I<nth>.  Returns an iterator to return items from the input
iterator, with a probability of C<1/$n> for each.  On average, in the
long run, 1 of every C<$n> items will be returned.

This can be useful for random sampling of data.

=item ifibonacci

 $iter = ifibonacci ();
 $iter = ifibonacci ($start);
 $iter = ifibonacci ($start1, $start2);

Generates a Fibonacci sequence.  If starting values are not specified,
uses (1, 1).  If only one is specified, it is used for both starting
values.

=item igeometric

 $iter = igeometric ($start, $end, $multiplier)

Generates a geometric sequence.  If C<$end> is undefined, the sequence
is unbounded.

I<Examples:>

 $iter = igeometric (1, 27, 3);         # 1, 3, 9, 27.
 $iter = igeometric (1, undef, 3);      # 1, 3, 9, 27, 81, ...
 $iter = igeometric (10, undef, 0.1);   # 10, 1, 0.1, 0.01, ...

=item imesh

=item izip

 $iter = imesh ($iter1, $iter2, ...);

This iterator accepts any number of other iterators, and "meshes"
their values together.  First it returns the first value of the first
iterator, then the first value of the second iterator, and so on,
until it has returned the first value of all of its iterator
arguments.  Then it goes back and returns the second value of the
first iterator, and so on.  It stops when any of its iterator
arguments is exhausted.

I<Example:>

 $i1 = ilist ('a', 'b', 'c');
 $i2 = ilist (1, 2, 3);
 $i3 = ilist ('rock', 'paper', 'scissors');
 $iter = imesh ($i1, $i2, $i3);
 # $iter will return, in turn, 'a', 1, 'rock', 'b', 2, 'paper', 'c',...

C<izip> is a synonym for C<imesh>.

=item iuniq

 $iter = iuniq ($another_iterator);

Creates an iterator to return unique values from another iterator;
weeds out duplicates.

I<Example:>

 $iter = ilist (1, 2, 2, 3, 1, 4);
 $uniq = iuniq ($iter);            # returns 1, 2, 3, 4.

=back

=head1 THINKING IN ITERATORS

Typically, when people approach a problem that involves manipulating a
bunch of data, their first thought is to load it all into memory, into
an array, and work with it in-place.  If you're only dealing with one
element at a time, this approach usually wastes memory needlessly.

For example, one might get a list of files to operate on, and loop
over it:

    my @files = fetch_file_list(....);
    foreach my $file (@files)
        ...
If fetch_file_list were modified to return an iterator instead of
an array, the same code could look like this:

    my $file_iterator = fetch_file_list(...)
    while ($file_iterator->isnt_exhausted)
        ...

The advantage here is that the whole list does not take up memory
while each individual element is being worked on.  For a list of
files, that's probably not a lot of overhead.  For the contents of
a file, on the other hand, it could be huge.

If a function requires a list of items as its input, the overhead
is tripled:

    sub myfunc
    {
        my @things = @_;
        ...

Now in addition to the array in the calling code, Perl must copy that
array to C<@_>, and then copy it again to C<@things>.  If you need to
massage the input from somewhere, it gets even worse:

    my @data = get_things_from_somewhere();
    my @filtered_data = grep {code} @data;
    my @transformed_data = map {code} @filtered_data;
    myfunc (@transformed_data);

If C<myfunc> is rewritten to use an Iterator instead of an array,
things become much simpler:

    my $data = ilist (get_things_from_somewhere());
    $filtered_data = igrep {code} $data;
    $transformed_data = imap {code} $filtered_data;
    myfunc ($transformed_data);

(This example assumes that the C<get_things_from_somewhere> function
cannot be modified to return an Iterator.  If it can, so much the
better!)  Now the original list is still in memory, inside the
C<$data> Iterator, but everwhere else, there is only one data element
in memory at a time.

Another advantage of Iterators is that they're homogeneous.  This is
useful for uncoupling library code from application code.  Suppose you
have a library function that grabs data from a filehandle:

    sub my_lib_func
    {
        my $fh = shift;
        ...

If you need C<my_lib_func> to get its data from a different source,
you must either modify it, or make a new copy of it that gets its
input differently, or you must jump through hoops to make the new
input stream look like a Perl filehandle.

On the other hand, if C<my_lib_func> accepts an iterator, then you
can pass it data from a filehandle:

    my $data = ifile "my_input.txt";
    $result = my_lib_func($data);

Or a database handle:

    my $data = imap {$_->{IMPORTANT_COLUMN}}
               idb_rows($dbh, 'select IMPORTANT_COLUMN from foo');
    $result = my_lib_func($data);

If you later decide you need to transform the data, or process only
every 10th data row, or whatever:

    $result = my_lib_func(imap {magic($_)} $data);
    $result = my_lib_func(inth 10, $data);

The library function doesn't care.  All it needs is an iterator.

Chapter 4 of Dominus's book (See L<SEE ALSO>) covers this topic in
some detail.

=head1 TUTORIAL

Let's create a date iterator.  It'll take a DateTime object as a
starting date, and return successive days -- that is, it'll add 1 day
each iteration.  It would be used as follows:

 use DateTime;

 $iter = I<(...something...);>
 $day1 = $iter->value;           # Initial date
 $day2 = $iter->value;           # One day later
 $day3 = $iter->value;           # Two days later

The easiest way to create such an iterator is by using a I<closure>.
If you're not familiar with the concept, it's fairly simple: In Perl,
the code within an I<anonymous block> has access to all the I<lexical
variables> that were in scope at the time the block was created.
After the program then leaves that lexical scope, those lexical
variables remain accessible by that code block for as long as it
exists.

This makes it very easy to create iterators that maintain their own
state.  Here we'll create a lexical scope by using a pair of braces:

 my $iter;
 {
    my $dt = DateTime->now();
    $iter = Iterator->new( sub
    {
        my $return_value = $dt->clone;
        $dt->add(days => 1);
        return $return_value;
    });
}

Because C<$dt> is lexically scoped to the outermost block, it is not
addressable from any code elsewhere in the program.  But the anonymous
block within the C<new> method's parentheses I<can> see C<$dt>.  So
C<$dt> does not get garbage collected as long as C<$iter> contains a
reference to it.

The code within the anonymous block is simple.  A copy of the current
C<$dt> is made, one day is added to C<$dt>, and the copy is returned.

You'll probably want to encapsulate the above block in a subroutine,
so that you could call it from anywhere in your program:

 sub date_iterator
 {
     my $dt = DateTime->now();
     return Iterator->new( sub
     {
         my $return_value = $dt->clone;
         $dt->add(days => 1);
         return $return_value;
     });
 }

If you look at the source code in Iterator.pm, you'll see that just
about all of the functions that create iterators look very similar to
the above C<date_iterator> function.

Of course, you'd probably want to be able to pass arguments to
C<date_iterator>, say a starting date, maybe an increment other than
"1 day".  But the basic idea is the same.

The above date iterator is an infinite (well, unbounded) iterator.
Let's look at how to indicate that your iterator has reached the end
of its sequence of values.  Let's write a scaled-down version of
C<irange> -- one that takes a start value and an end value and always
increments by 1.

 sub irange_limited
 {
     my ($start, $end) = @_;

     return Iterator->new (sub
     {
         Iterator::is_done
             if $start > $end;

         return $start++;
     });
 }

The iterator itself is very simple (this sort of thing gets to be easy
once you get the hang of it).  The new element here is the signalling
that the sequence has ended, and the iterator's work is done.
C<is_done> is how your code indicates this to the Iterator object.

You may also want to throw an exception if the user specified bad input
parameters.  There are a couple ways you can do this.

     ...
     die "Too few parameters to irange_limited"  if @_ < 2;
     die "Too many parameters to irange_limited" if @_ > 2;
     my ($start, $end) = @_;
     ...

This is the simplest way; you just use C<die> (or C<croak>).  You may
choose to throw an Iterator parameter error, though; this will make
your function work more like one of Iterator.pm's built in functions:

     ...
     Iterator::X::Parameter_Error->throw(
         "Too few parameters to irange_limited")
         if @_ < 2;
     Iterator::X::Parameter_Error->throw(
         "Too many parameters to irange_limited")
         if @_ > 2;
     my ($start, $end) = @_;
     ...


=head1 EXPORTS

No symbols are exported to the caller's namespace by default.

Various sets of functions may be optionally exported:

 :util
        imap igrep irange ilist iarray ihead iappend ipairwise
        iskip skip_until imesh izip iuniq

 :file
        idir_listing idir_walk ifile ifile_reverse

 :dbi
        idb_rows

 :misc
        ipermute inth irand_nth ifibonacci igeometric

=head1 DIAGNOSTICS

Iterator uses Exception::Class objects for throwing exceptions.
If you're not familiar with Exception::Class, don't worry; these
exception objects work just like C<$@> does with C<die> and C<croak>,
but they are easier to work with if you are trapping errors.

All exceptions thrown by Iterator have a base class of Iterator::X.
You can trap errors with an eval block:

 eval { $foo = $iterator->value(); };

and then check for errors as follows:

 if (Iterator::X->caught())  {...

You can look for more specific errors by looking at a more specific
class:

 if (Iterator::X::Exhausted->caught())  {...

Some exceptions may provide further information, which may be useful
for your exception handling:

 if (my $ex = Iterator::X::User_Code_Error->caught())
 {
     my $exception = $ex->eval_error();
     ...

If you choose not to (or cannot) handle a particular type of exception
(for example, there's not much to be done about a parameter error),
you should rethrow the error:

 if (my $ex = Iterator::X->caught())
 {
     if ($ex->isa('Iterator::X::Something_Useful'))
     {
         ...
     }
     else
     {
         $ex->rethrow();
     }
 }

=over 4

=item * Parameter Errors

Class: C<Iterator::X::Parameter_Error>

You called an Iterator method with one or more bad parameters.  Since
this is almost certainly a coding error, there is probably not much
use in handling this sort of exception.

As a string, this exception provides a human-readable message about
what the problem was.

=item * Exhausted Iterators

Class: C<Iterator::X::Exhausted>

You called C<value> on an iterator that is exhausted; that is, there
are no more values in the sequence to return.

As a string, this exception is "Iterator is exhausted."

=item * End of Sequence

Class: C<Iterator::X::Am_Now_Exhausted>

This exception is not thrown directly by any Iterator.pm methods, but
is to be thrown by iterator sequence generation code; that is, the
code that you pass to the C<new> constructor.  Your code won't catch
an C<Am_Now_Exhausted> exception, because the Iterator object will
catch it internally and set its C<is_exhausted> flag.

The simplest way to throw this exception is to use the C<is_done>
function:

 Iterator::is_done() if $something;

=item * User Code Exceptions

Class: C<Iterator::X::User_Code_Error>

This exception is thrown when the sequence generation code throws any
sort of error besides C<Am_Now_Exhausted>.  This could be because your
code explicitly threw an error (that is, C<die>d), or because it
otherwise encountered an exception (any runtime error).

This exception has one method, C<eval_error>, which returns the
original C<$@> that was trapped by the Iterator object.  This may be a
string or an object, depending on how C<die> was invoked.

As a string, this exception evaluates to the stringified C<$@>.

=item * I/O Errors

Class: C<Iterator::X::IO_Error>

This exception is thrown when any sort of I/O error occurs; this
only happens with the filesystem iterators.

This exception has one method, C<os_error>, which returns the original
C<$!> that was trapped by the Iterator object.

As a string, this exception provides some human-readable information
along with C<$!>.

=item * Internal Errors

Class: C<Iterator::X::Internal_Error>

Something happened that I thought couldn't possibly happen.  I would
appreciate it if you could send me an email message detailing the
circumstances of the error.

=back

=head1 REQUIREMENTS

Requires the following additional modules:

C<Exception::Class>, v1.21 or later.

C<IO::Dir> and C<Cwd> are required if you use idir_listing or dir_walk.

C<IO::File> is required if you use ifile or ifile_reverse

=head1 SEE ALSO

I<Higher Order Perl>, Mark Jason Dominus, Morgan Kauffman 2005.

 http://perl.plover.com/hop/

=head1 THANKS

Much thanks to Will Coleda and Paul Lalli (and the RPI lily crowd in
general) for suggestions for the pre-release version.

=head1 AUTHOR / COPYRIGHT

Eric J. Roode, roode@cpan.org

Copyright (c) 2005 by Eric J. Roode.  All Rights Reserved.
This module is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

To avoid my spam filter, please include "Perl", "module", or this
module's name in the message's subject line, and/or GPG-sign your
message.

If you have suggestions for improvement, please drop me a line.  If
you make improvements to this software, I ask that you please send me
a copy of your changes. Thanks.

=cut

=begin gpg

-----BEGIN PGP SIGNATURE-----
Version: GnuPG v1.4.1 (Cygwin)

iD8DBQFC7+3HY96i4h5M0egRAt2yAKD2dRrpAFgn2HFauPKx4mqmECEiRgCgyKJe
+bsJk8jQooi1GkB6UArK6Sg=
=AoOH
-----END PGP SIGNATURE-----

=end gpg
